<?php

namespace BalletMecanique\PianolaLaravel\Http\Controllers;

use BalletMecanique\PianolaLaravel\Models\SharedDataList;
use BalletMecanique\PianolaLaravel\Models\SharedDataListMember;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class SharedDataListController
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $user = auth()->user();
        if (! $user) {
            return [];
        }
        $baseTable = $request->query('base_table');

        $query = SharedDataList::with(['members'])
            ->where('user_id', $user->id);

        if ($baseTable) {
            $query->where('base_table', $baseTable);
        }

        $sharedDataLists = $query->get();

        // Transform to only return id, name, and membersCommaSeparated
        $result = $sharedDataLists->map(function ($list) {
            return [
                'id' => $list->id,
                'name' => $list->name,
                'expiration_date' => $list->expiration_date,
                'membersCommaSeparated' => $list->membersCommaSeparated,
            ];
        });

        return response()->json($result);
    }

    /**
     * Display the specified resource.
     */
    public function show($id, Request $request)
    {
        $user = auth()->user();
        if (! $user) {
            return response()->json(['error' => 'Unauthorized'], 401);
        }

        $sharedDataList = SharedDataList::with(['members'])
            ->where('user_id', $user->id)
            ->findOrFail($id);

        return response()->json($sharedDataList);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'base_table' => 'required|string|max:255',
            'pianola_list_id' => 'nullable|integer|exists:pianola_lists,id',
            'pianola_query_id' => 'nullable|integer|exists:pianola_queries,id',
            'expiration_date' => 'nullable|date',
            'filter_field' => 'nullable|string|max:255',
            'filter_value' => 'nullable|string|max:255',
            'members' => 'nullable|array',
            'members.*' => 'email',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        $data = $request->only([
            'name',
            'base_table',
            'pianola_list_id',
            'pianola_query_id',
            'expiration_date',
            'filter_field',
            'filter_value',
        ]);

        $data['user_id'] = Auth::id();

        $sharedDataList = SharedDataList::create($data);

        // Handle members
        if ($request->has('members') && is_array($request->members)) {
            $this->syncMembers($sharedDataList, $request->members);
        }

        return response()->json($sharedDataList->load('members'), 201);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update($id, Request $request)
    {
        $user = Auth::user();
        if (! $user) {
            return response()->json(['error' => 'Unauthorized'], 401);
        }

        $sharedDataList = SharedDataList::where('user_id', $user->id)->findOrFail($id);

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'base_table' => 'required|string|max:255',
            'pianola_list_id' => 'nullable|integer|exists:pianola_lists,id',
            'pianola_query_id' => 'nullable|integer|exists:pianola_queries,id',
            'expiration_date' => 'nullable|date',
            'filter_field' => 'nullable|string|max:255',
            'filter_value' => 'nullable|string|max:255',
            'members' => 'nullable|array',
            'members.*' => 'email',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        $data = $request->only([
            'name',
            'base_table',
            'pianola_list_id',
            'pianola_query_id',
            'expiration_date',
            'filter_field',
            'filter_value',
        ]);

        $sharedDataList->update($data);

        // Handle members
        if ($request->has('members') && is_array($request->members)) {
            $this->syncMembers($sharedDataList, $request->members);
        }

        return response()->json($sharedDataList->load('members'));
    }

    /**
     * Sync members for a shared data list.
     * Compares current members with new members array and adds/removes as needed.
     */
    private function syncMembers(SharedDataList $sharedDataList, array $newEmails)
    {
        // Get current member emails
        $currentEmails = $sharedDataList->members->pluck('email')->toArray();

        // Find emails to add (in new but not in current)
        $emailsToAdd = array_diff($newEmails, $currentEmails);

        // Find emails to remove (in current but not in new)
        $emailsToRemove = array_diff($currentEmails, $newEmails);

        // Add new members
        foreach ($emailsToAdd as $email) {
            SharedDataListMember::create([
                'pianola_shared_data_list_id' => $sharedDataList->id,
                'email' => $email,
            ]);
        }

        // Remove old members
        if (! empty($emailsToRemove)) {
            SharedDataListMember::where('pianola_shared_data_list_id', $sharedDataList->id)
                ->whereIn('email', $emailsToRemove)
                ->delete();
        }
    }

    public function previewUrl($id, Request $request)
    {
        // Get the shared data list to ensure it exists
        $sharedDataList = SharedDataList::findOrFail($id);

        // Get the configured unlock token
        $unlockToken = config('pianola-laravel.shared_data_unlock_token');

        if (!$unlockToken) {
            return response()->json([
                'error' => 'Unlock token not configured',
                'message' => 'The shared data unlock token is not configured in the application.'
            ], 500);
        }

        // Build the full URL using the same logic as PianolaLaravelServiceProvider
        $subdomain = config('pianola-laravel.shared_data_subdomain', 'data-share');
        $rootDomain = $this->getRootDomain();
        $protocol = $request->isSecure() ? 'https' : 'http';

        $previewUrl = $protocol . '://' . $subdomain . '.' . $rootDomain . '/' . $sharedDataList->id . '?unlock_token=' . urlencode($unlockToken);

        return response()->json([
            'preview_url' => $previewUrl,
            'shared_data_list' => [
                'id' => $sharedDataList->id,
                'name' => $sharedDataList->name,
            ]
        ]);
    }

    /**
     * Get the root domain (same logic as PianolaLaravelServiceProvider)
     */
    protected function getRootDomain(): string
    {
        $host = request()?->getHost();
        $hostParts = explode('.', $host);

        // If we have more than 2 parts (subdomain.domain.tld), get the last 2 parts
        if (count($hostParts) > 2) {
            return implode('.', array_slice($hostParts, -2));
        }

        return $host;
    }
}
