<?php

namespace BalletMecanique\PianolaLaravel;

use BalletMecanique\PianolaLaravel\Commands\CreateDevUserCommand;
use BalletMecanique\PianolaLaravel\Commands\CreateModelsCommand;
use BalletMecanique\PianolaLaravel\Commands\GenerateFileUploadBoilerplateCommand;
use BalletMecanique\PianolaLaravel\Commands\PianolaInstallCommand;
use BalletMecanique\PianolaLaravel\Commands\SendAlertsCommand;
use BalletMecanique\PianolaLaravel\Commands\UpdateSchemaCommand;
use BalletMecanique\PianolaLaravel\Http\Controllers\ConfigController;
use BalletMecanique\PianolaLaravel\Http\Controllers\UserController;
use Illuminate\Database\Eloquent\Relations\Relation;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Route;
use Illuminate\Support\Str;
use Laravel\Fortify\Http\Controllers\AuthenticatedSessionController;
use Laravel\Fortify\Http\Controllers\TwoFactorAuthenticatedSessionController;
use Laravel\Fortify\Http\Controllers\TwoFactorAuthenticationController;
use Laravel\Fortify\Http\Controllers\TwoFactorQrCodeController;
use Laravel\Sanctum\Http\Controllers\CsrfCookieController;
use Spatie\LaravelPackageTools\Package;
use Spatie\LaravelPackageTools\PackageServiceProvider;

class PianolaLaravelServiceProvider extends PackageServiceProvider
{
    public function configurePackage(Package $package): void
    {
        /*
         * This class is a Package Service Provider
         *
         * More info: https://github.com/spatie/laravel-package-tools
         */
        $package
            ->name('pianola-laravel')
            ->hasConfigFile()
            ->hasViews()
            ->hasMigrations([
                'create_countries_table',
                'create_people_table',
                'create_companies_table',
                'create_employees_table',
            ])
            ->hasCommand(PianolaInstallCommand::class)
            ->hasCommand(CreateModelsCommand::class)
            ->hasCommand(UpdateSchemaCommand::class)
            ->hasCommand(CreateDevUserCommand::class)
            ->hasCommand(SendAlertsCommand::class)
            ->hasCommand(GenerateFileUploadBoilerplateCommand::class);
    }

    public function packageBooted()
    {
        if ($this->app->runningInConsole()) {
            // Publish assets
            $this->publishes([
                __DIR__.'/../resources/assets' => public_path('pianola-laravel'),
            ], 'pianola-laravel-assets');
        }
        $this->publishes([
            __DIR__.'/../config/pianola-laravel.php' => base_path('config/pianola.php'),
        ], 'config');
        $this->publishes([
            __DIR__.'/../boilerplate/pianola-routes.php' => base_path('routes/pianola.php'),
        ], 'pianola-routes');
        $this->publishes([
            __DIR__.'/../boilerplate/create_files_table.php' => base_path('database/migrations/2023_01_31_173300_create_files_table.php'),
            __DIR__.'/../boilerplate/create_file_tags_table.php' => base_path('database/migrations/2023_01_31_173400_create_file_tags_table.php'),
        ], 'file-migrations');
        $this->publishes([
            __DIR__.'/../boilerplate/File.php' => base_path('app/models/File.php'),
            __DIR__.'/../boilerplate/FileTag.php' => base_path('app/models/FileTag.php'),
        ], 'file-models');
        $this->publishes([
            __DIR__.'/../boilerplate/FileUploadController.php' => base_path('app/Http/Controllers/FileUploadController.php'),
        ], 'file-upload-controller');
        $this->publishes([
            __DIR__.'/../boilerplate/FileUploadControllerVapor.php' => base_path('app/Http/Controllers/FileUploadController.php'),
            __DIR__.'/../boilerplate/SignedStorageUrlController.php' => base_path('app/Http/Controllers/SignedStorageUrlController.php'),
        ], 'file-upload-controllers-vapor');

        Relation::enforceMorphMap($this->getMorphMapForAllSchemaTables());
    }

    public function packageRegistered()
    {
        //api routes
        Route::get('api/pianola/login-config', [ConfigController::class, 'getLoginConfig']);
        //TO DO: remove these when front-end app is adjusted
        Route::group(['middleware' => 'web'], function () {
            Route::post('api/login', [AuthenticatedSessionController::class, 'store'])->name('login');
            Route::post('api/logout', [AuthenticatedSessionController::class, 'destroy'])->name('logout');
            Route::get('api/sanctum/csrf-cookie', CsrfCookieController::class.'@show');
            //2FA Routes
            Route::get('api/user', [UserController::class, 'returnUser']);
            Route::post('api/user/', [TwoFactorAuthenticationController::class, 'store']);
            Route::post('api/user/two-factor-authentication', [TwoFactorAuthenticationController::class, 'store']);
            Route::get('api/user/two-factor-qr-code', [TwoFactorQrCodeController::class, 'show']);
            Route::post('api/two-factor-challenge', [TwoFactorAuthenticatedSessionController::class, 'store']);
        });
        //EOTODO

        //pianola api routes
        Route::group($this->pianolaApiRouteConfiguration(), function () {
            //register specific package routes
            $this->loadRoutesFrom(__DIR__.'/../routes/pianola.php');
            //register app routes (which can override specific package routes)
            if (File::exists(base_path('/routes/pianola.php'))) {
                $this->loadRoutesFrom(base_path('/routes/pianola.php'));
            }
            //register fallback routes
            $this->loadRoutesFrom(__DIR__.'/../routes/pianolaFallback.php');
        });
        //web routes
        $this->loadRoutesFrom(__DIR__.'/../routes/web.php');
        //load migrations
        $this->loadMigrationsFrom(__DIR__.'/../database/package_migrations');
    }

    protected function pianolaApiRouteConfiguration()
    {
        return [
            'prefix' => 'api/pianola',
            'middleware' => ['web', 'auth'],
        ];
    }

    protected function getMorphMapForAllSchemaTables()
    {
        if (! File::exists(config_path('/pianola/schema.json'))) {
            return [];
        }
        $schema = File::get(config_path('/pianola/schema.json'));
        $schema = json_decode($schema, true);
        $morphMap = [];
        foreach ($schema as $table) {
            $tableName = $table['name'];
            $modelName = Str::singular(Str::studly($tableName));
            $morphMap[$tableName] = 'App\\Models\\'.$modelName;
        }

        return $morphMap;
    }
}
