<?php

namespace BalletMecanique\PianolaLaravel\Http\Middleware;

use BalletMecanique\PianolaLaravel\Models\SharedDataUser;
use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;

class SharedDataApiKeyMiddleware
{
    /**
     * Handle an incoming request.
     *
     * @param  \Closure(\Illuminate\Http\Request): (\Illuminate\Http\Response|\Illuminate\Http\RedirectResponse)  $next
     * @return \Illuminate\Http\Response|\Illuminate\Http\RedirectResponse
     */
    public function handle(Request $request, Closure $next)
    {
        // Check for API key in header or query parameter
        $apiKey = $request->header('X-API-Key') ?? $request->get('api_key');

        if (!$apiKey) {
            return response()->json([
                'error' => 'API key required',
                'message' => 'Please provide an API key via X-API-Key header or api_key parameter'
            ], 401);
        }

        // Find user by API key
        $user = SharedDataUser::where('api_key', $apiKey)->first();

        if (!$user) {
            Log::warning('Invalid API key provided', [
                'api_key' => substr($apiKey, 0, 10) . '...',
                'ip' => $request->ip(),
                'user_agent' => $request->userAgent(),
            ]);

            return response()->json([
                'error' => 'Invalid API key',
                'message' => 'The provided API key is not valid'
            ], 401);
        }

        // Log successful API access
        Log::info('API access granted', [
            'user_id' => $user->id,
            'user_email' => $user->email,
            'endpoint' => $request->path(),
            'ip' => $request->ip(),
        ]);

        // Set the authenticated user for this request
        Auth::guard('shared_data')->setUser($user);

        return $next($request);
    }
}
