<?php

namespace BalletMecanique\PianolaLaravel\Http\Controllers;

use BalletMecanique\PianolaLaravel\Models\SharedDataUser;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;

class SharedDataAuthController
{
    /**
     * Show the registration form
     */
    public function showRegistrationForm()
    {
        // Get app configuration
        $appConfig = $this->getAppConfig();

        // Set app locale
        app()->setLocale($appConfig['appLanguage']);

        return view('pianola-laravel::shared-data.register', compact('appConfig'));
    }

    /**
     * Handle registration
     */
    public function register(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:pianola_shared_data_users',
            'password' => 'required|string|min:8|confirmed',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'message' => 'Validation failed',
                'errors' => $validator->errors(),
            ], 422);
        }

        $user = SharedDataUser::create([
            'name' => $request->name,
            'email' => $request->email,
            'password' => Hash::make($request->password),
        ]);

        // Log the user in
        Auth::guard('shared_data')->login($user);

        return response()->json([
            'message' => 'Registration successful',
            'user' => $user,
        ], 201);
    }

    /**
     * Show the login form
     */
    public function showLoginForm()
    {
        // Get app configuration
        $appConfig = $this->getAppConfig();

        // Set app locale
        app()->setLocale($appConfig['appLanguage']);

        return view('pianola-laravel::shared-data.login', compact('appConfig'));
    }

    /**
     * Handle login
     */
    public function login(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|email',
            'password' => 'required',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'message' => 'Validation failed',
                'errors' => $validator->errors(),
            ], 422);
        }

        $credentials = $request->only('email', 'password');

        if (Auth::guard('shared_data')->attempt($credentials, $request->boolean('remember'))) {
            $request->session()->regenerate();

            return response()->json([
                'message' => 'Login successful',
                'user' => Auth::guard('shared_data')->user(),
            ]);
        }

        return response()->json([
            'message' => 'Invalid credentials',
        ], 401);
    }

    /**
     * Handle logout
     */
    public function logout(Request $request)
    {
        Auth::guard('shared_data')->logout();
        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return response()->json([
            'message' => 'Logout successful',
        ]);
    }

    /**
     * Get the current authenticated user
     */
    public function user()
    {
        return response()->json([
            'user' => Auth::guard('shared_data')->user(),
        ]);
    }

    /**
     * Get app configuration from pianola/app.json
     */
    protected function getAppConfig()
    {
        if (! File::exists(config_path('pianola/app.json'))) {
            return [
                'colorPrimary' => '#007cba',
                'logoPath' => null,
                'appName' => 'Shared Data Portal',
                'appLanguage' => 'en',
            ];
        }

        $config = json_decode(File::get(config_path('pianola/app.json')), true);
        $basicConfig = $config['basicConfig'] ?? [];

        return [
            'colorPrimary' => $basicConfig['colorPrimary'] ?? '#007cba',
            'logoPath' => $basicConfig['logoPath'] ?? null,
            'appName' => $basicConfig['appName'] ?? 'Shared Data Portal',
            'appLanguage' => $basicConfig['appLanguage'] ?? 'en',
            'backgroundPath' => $basicConfig['backgroundPath'] ?? null,
        ];
    }
}
