<?php

namespace BalletMecanique\PianolaLaravel\Http\Controllers;

use BalletMecanique\PianolaLaravel\Models\SharedDataList;
use BalletMecanique\PianolaLaravel\Models\SharedDataListMember;
use BalletMecanique\PianolaLaravel\Services\QueryBuilder;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Str;

class SharedDataController
{
    /**
     * Display a listing of shared data
     */
    public function index(Request $request)
    {
        $user = Auth::guard('shared_data')->user();

        // Determine which lists to show based on user type
        if ($user?->email === 'test@example.com') {
            // Test user sees all lists (excluding expired ones)
            $lists = SharedDataList::with(['members', 'pianolaList', 'pianolaQuery', 'user'])
                ->where(function ($query) {
                    $query->whereNull('expiration_date')
                        ->orWhere('expiration_date', '>=', now()->toDateString());
                })
                ->get();
        } else {
            // Regular users see only lists they're members of (excluding expired ones)
            $listIds = SharedDataListMember::where('email', $user?->email)
                ->pluck('pianola_shared_data_list_id');

            $lists = SharedDataList::with(['members', 'pianolaList', 'pianolaQuery', 'user'])
                ->whereIn('id', $listIds)
                ->where(function ($query) {
                    $query->whereNull('expiration_date')
                        ->orWhere('expiration_date', '>=', now()->toDateString());
                })
                ->get();
        }

        // Filter lists to only show those where the creator has data sharing rights for the module
        $lists = $lists->filter(function ($list) {
            return $this->userCanShareDataForModule($list->user, $list->base_table);
        });

        // Check if this is an API request
        if ($request->expectsJson() || $request->is('api/*')) {
            // Return JSON for API requests
            return response()->json([
                'message' => 'Shared data index',
                'user' => $user,
                'lists' => $lists,
            ]);
        }

        // Get app configuration
        $appConfig = $this->getAppConfig();

        // Set app locale
        app()->setLocale($appConfig['appLanguage']);

        // Return view for web requests
        return view('pianola-laravel::shared-data.index', compact('lists', 'user', 'appConfig'));
    }

    /**
     * Display the specified shared data item
     */
    public function show(Request $request, $id)
    {
        $user = Auth::guard('shared_data')->user();

        // Find the shared data list
        $sharedDataList = SharedDataList::with(['pianolaList', 'pianolaQuery', 'user'])->findOrFail($id);

        // Check if the list has expired
        if ($sharedDataList->expiration_date && $sharedDataList->expiration_date < now()->toDateString()) {
            abort(404, 'This shared data list has expired.');
        }

        // Check if the creator has permission to share data for this module
        if (!$this->userCanShareDataForModule($sharedDataList->user, $sharedDataList->base_table)) {
            abort(403, 'This shared data list is no longer available due to permission restrictions.');
        }

        // Check access permissions (same logic as index)
        if ($user?->email !== 'test@example.com') {
            $hasAccess = SharedDataListMember::where('pianola_shared_data_list_id', $id)
                ->where('email', $user?->email)
                ->exists();

            if (! $hasAccess) {
                abort(403, 'You do not have access to this shared data list.');
            }
        }

        // Get the data using QueryBuilder
        $data = [];
        $columns = [];

        if ($sharedDataList->pianolaQuery && $sharedDataList->base_table) {
            // Create the model instance
            $modelName = 'App\\Models\\'.Str::singular(Str::studly($sharedDataList->base_table));

            if (class_exists($modelName)) {
                $model = new $modelName;

                // Prepare request parameters for QueryBuilder
                $queryRequest = [
                    'q' => $sharedDataList->pianolaQuery->query,
                    'page' => 'all', // Get all results for shared data
                ];

                // Add constant pre-filter if filter_field and filter_value are set
                if ($sharedDataList->filter_field && $sharedDataList->filter_value) {
                    $queryRequest['c'] = $sharedDataList->filter_field.'|'.$sharedDataList->filter_value;
                }

                // Add default sort order if available
                $defaultSort = $this->getDefaultSort($sharedDataList->pianolaList);
                if ($defaultSort) {
                    $queryRequest['sort'] = $defaultSort;
                }

                // Execute the query
                $queryBuilder = new QueryBuilder($model, $queryRequest);
                $data = $queryBuilder->get();
            }
        }

        // Get column configuration from PianolaList
        if ($sharedDataList->pianolaList && $sharedDataList->pianolaList->selection) {
            $selection = json_decode($sharedDataList->pianolaList->selection, true);

            // Filter out selection and action columns
            $columns = collect($selection)->filter(function ($column) {
                $text = strtolower($column['text'] ?? '');
                $value = strtolower($column['value'] ?? '');

                return ! in_array($text, ['selection', 'action']) && ! in_array($value, ['selection', 'action']);
            })->values()->toArray();
        }

        // Check if this is an API request
        if ($request->expectsJson() || $request->is('api/*')) {
            return response()->json([
                'shared_data_list' => $sharedDataList,
                'data' => $data,
                'columns' => $columns,
                'user' => $user,
            ]);
        }

        // Get app configuration
        $appConfig = $this->getAppConfig();

        // Set app locale
        app()->setLocale($appConfig['appLanguage']);

        // Check if calendar is available for this module
        $hasCalendar = $this->getHasCalendar($sharedDataList->pianolaList);

        // Check display mode
        $displayMode = $request->get('display', 'list');

        if ($displayMode === 'calendar' && $hasCalendar) {
            // Get calendar data mapping (truncate timestamps for calendar display)
            $calendarData = $this->getCalendarData($sharedDataList->pianolaQuery, $data, true);

            // For calendar view, only pass the record count instead of full data to reduce response size
            $recordCount = $data->count();

            return view('pianola-laravel::shared-data.calendar', compact('sharedDataList', 'recordCount', 'columns', 'user', 'appConfig', 'hasCalendar', 'calendarData'));
        }

        if ($displayMode === 'calendar_print' && $hasCalendar) {
            // Get calendar data for printing
            $calendarData = $this->getCalendarData($sharedDataList->pianolaQuery, $data, true);

            // Get print parameters
            $type = $request->get('type', 'daily'); // daily or weekly
            $date = $request->get('date', now()->format('Y-m-d'));

            // Filter events for the specified date/period
            $events = $this->filterEventsForPrint($calendarData, $type, $date);

            // Get locations for the filtered events
            $locations = $this->getLocationsForPrint($events);

            // Get app configuration for logo and title
            $logoPath = $appConfig['logoPath'] ?? null;
            $title = $this->getPrintTitle($date, $type);

            return view('pianola-laravel::printouts.calendar', compact('events', 'date', 'locations', 'title', 'logoPath', 'type'));
        }

        if ($displayMode === 'calendar_print_all' && $hasCalendar) {
            // Get calendar data for printing all periods
            $calendarData = $this->getCalendarData($sharedDataList->pianolaQuery, $data, true);

            // Get print parameters
            $type = $request->get('type', 'daily'); // daily or weekly

            // Get all available periods and their data
            $allPeriodsData = $this->getAllPeriodsForPrint($calendarData, $type);

            // Get app configuration for logo and title
            $logoPath = $appConfig['logoPath'] ?? null;
            $listName = $sharedDataList->name;

            // Set app locale for translations
            app()->setLocale($appConfig['appLanguage']);

            return view('pianola-laravel::printouts.calendar-all', compact('allPeriodsData', 'logoPath', 'type', 'listName', 'appConfig'));
        }

        // For table view, pre-filter data to only include columns that will be displayed
        // This reduces response size significantly by removing unused database fields
        $filteredData = $this->filterDataByColumns($data, $columns);
        $recordCount = $data->count();

        // Return default list view
        return view('pianola-laravel::shared-data.show', compact('sharedDataList', 'filteredData', 'recordCount', 'columns', 'user', 'appConfig', 'hasCalendar'));
    }

    /**
     * API endpoint for shared data
     */
    public function api(Request $request, $id)
    {
        $user = Auth::guard('shared_data')->user();

        // Find the shared data list
        $sharedDataList = SharedDataList::with(['pianolaList', 'pianolaQuery', 'user'])->findOrFail($id);

        // Check if the list has expired
        if ($sharedDataList->expiration_date && $sharedDataList->expiration_date < now()->toDateString()) {
            return response()->json([
                'error' => 'Expired',
                'message' => 'This shared data list has expired.',
            ], 404);
        }

        // Check if the creator has permission to share data for this module
        if (!$this->userCanShareDataForModule($sharedDataList->user, $sharedDataList->base_table)) {
            return response()->json([
                'error' => 'Permission denied',
                'message' => 'This shared data list is no longer available due to permission restrictions.',
            ], 403);
        }

        // Check access permissions (same logic as show method)
        if ($user?->email !== 'test@example.com') {
            $hasAccess = SharedDataListMember::where('pianola_shared_data_list_id', $id)
                ->where('email', $user?->email)
                ->exists();

            if (! $hasAccess) {
                return response()->json([
                    'error' => 'Access denied',
                    'message' => 'You do not have access to this shared data list.',
                ], 403);
            }
        }

        // Get the data using the same logic as the show method
        $data = $this->getData($sharedDataList);
        $columns = $this->getColumns($sharedDataList);

        // Apply filtering if specified
        if ($sharedDataList->filter_field && $sharedDataList->filter_value) {
            $data = $data->where($sharedDataList->filter_field, $sharedDataList->filter_value);
        }

        // Convert to array and get only essential fields
        $filteredData = $data->map(function ($record) use ($columns) {
            $essentialData = [];
            foreach ($columns as $column) {
                $value = $this->getFieldValue($record, $column['value']);
                $essentialData[$column['value']] = $value;
            }

            return $essentialData;
        });

        return response()->json([
            'id' => $sharedDataList->id,
            'name' => $sharedDataList->name,
            'base_table' => $sharedDataList->base_table,
            'record_count' => $filteredData->count(),
            'columns' => $columns,
            'data' => $filteredData,
            'generated_at' => now()->toISOString(),
        ]);
    }

    /**
     * Generate an unlock token for a given email
     * This is a utility method that can be used to create unlock tokens
     */
    public function generateUnlockToken(Request $request)
    {
        $request->validate([
            'email' => 'required|email|exists:pianola_shared_data_users,email',
        ]);

        $token = Crypt::encryptString($request->email);

        return response()->json([
            'unlock_token' => $token,
        ]);
    }

    /**
     * Generate or regenerate API key for the authenticated user
     */
    public function generateApiKey(Request $request, $id)
    {
        $user = Auth::guard('shared_data')->user();

        // Find the shared data list to ensure user has access
        $sharedDataList = SharedDataList::with('user')->findOrFail($id);

        // Check if the creator has permission to share data for this module
        if (!$this->userCanShareDataForModule($sharedDataList->user, $sharedDataList->base_table)) {
            return response()->json([
                'error' => 'Permission denied',
                'message' => 'This shared data list is no longer available due to permission restrictions.',
            ], 403);
        }

        // Check access permissions (same logic as show method)
        if ($user?->email !== 'test@example.com') {
            $hasAccess = SharedDataListMember::where('pianola_shared_data_list_id', $id)
                ->where('email', $user?->email)
                ->exists();

            if (! $hasAccess) {
                return response()->json([
                    'error' => 'Access denied',
                    'message' => 'You do not have access to this shared data list.',
                ], 403);
            }
        }

        // Generate new API key
        $apiKey = $user?->generateApiKey();

        // Redirect back to the API management page
        return redirect()->route('shared-data.api-management', $id)
            ->with('success', 'API key generated successfully');
    }

    /**
     * Show API management screen
     */
    public function apiManagement(Request $request, $id)
    {
        $user = Auth::guard('shared_data')->user();

        // Find the shared data list
        $sharedDataList = SharedDataList::with('user')->findOrFail($id);

        // Check if the creator has permission to share data for this module
        if (!$this->userCanShareDataForModule($sharedDataList->user, $sharedDataList->base_table)) {
            abort(403, 'This shared data list is no longer available due to permission restrictions.');
        }

        // Check access permissions (same logic as show method)
        if ($user?->email !== 'test@example.com') {
            $hasAccess = SharedDataListMember::where('pianola_shared_data_list_id', $id)
                ->where('email', $user?->email)
                ->exists();

            if (! $hasAccess) {
                abort(403, 'You do not have access to this shared data list.');
            }
        }

        // Get app configuration
        $appConfig = $this->getAppConfig();

        // Set app locale
        app()->setLocale($appConfig['appLanguage']);

        // Generate API endpoint URL
        $subdomain = config('pianola-laravel.shared_data_subdomain', 'data-share');
        $rootDomain = $this->getRootDomain();
        $apiEndpointUrl = "https://{$subdomain}.{$rootDomain}/api/{$id}";

        // Generate sample API response
        $sampleResponse = $this->generateSampleApiResponse($sharedDataList);

        return view('pianola-laravel::shared-data.api-management', compact(
            'sharedDataList',
            'user',
            'appConfig',
            'apiEndpointUrl',
            'sampleResponse'
        ));
    }

    /**
     * Get data for a shared data list using QueryBuilder
     */
    protected function getData($sharedDataList)
    {
        $data = collect();

        if ($sharedDataList->pianolaQuery && $sharedDataList->base_table) {
            // Create the model instance
            $modelName = 'App\\Models\\'.Str::singular(Str::studly($sharedDataList->base_table));

            if (class_exists($modelName)) {
                $model = new $modelName;

                // Prepare request parameters for QueryBuilder
                $queryRequest = [
                    'q' => $sharedDataList->pianolaQuery->query,
                    'page' => 'all', // Get all results for shared data
                ];

                // Add constant pre-filter if filter_field and filter_value are set
                if ($sharedDataList->filter_field && $sharedDataList->filter_value) {
                    $queryRequest['c'] = $sharedDataList->filter_field.'|'.$sharedDataList->filter_value;
                }

                // Add default sort order if available
                $defaultSort = $this->getDefaultSort($sharedDataList->pianolaList);
                if ($defaultSort) {
                    $queryRequest['sort'] = $defaultSort;
                }

                // Execute the query
                $queryBuilder = new QueryBuilder($model, $queryRequest);
                $data = $queryBuilder->get();
            }
        }

        return $data;
    }

    /**
     * Get columns configuration for a shared data list
     */
    protected function getColumns($sharedDataList)
    {
        $columns = [];

        // Get column configuration from PianolaList
        if ($sharedDataList->pianolaList && $sharedDataList->pianolaList->selection) {
            $selection = json_decode($sharedDataList->pianolaList->selection, true);

            // Filter out selection and action columns
            $columns = collect($selection)->filter(function ($column) {
                $text = strtolower($column['text'] ?? '');
                $value = strtolower($column['value'] ?? '');

                return ! in_array($text, ['selection', 'action']) && ! in_array($value, ['selection', 'action']);
            })->values()->toArray();
        }

        return $columns;
    }

    /**
     * Get field value from record (handles nested fields and relationships)
     */
    protected function getFieldValue($record, $field)
    {
        if (is_array($record)) {
            return Arr::get($record, $field, '');
        }

        // Handle Eloquent models
        if (is_object($record)) {
            return $this->getNestedValue($record, $field);
        }

        return '';
    }

    /**
     * Get the root domain (copied from service provider)
     */
    protected function getRootDomain(): string
    {
        $host = request()?->getHost();
        $hostParts = explode('.', $host);

        // If we have more than 2 parts (subdomain.domain.tld), get the last 2 parts
        if (count($hostParts) > 2) {
            return implode('.', array_slice($hostParts, -2));
        }

        // Otherwise return the full host
        return $host;
    }

    /**
     * Generate a sample API response for preview
     */
    protected function generateSampleApiResponse($sharedDataList)
    {
        // Get sample data (first few records)
        $data = $this->getData($sharedDataList);
        $columns = $this->getColumns($sharedDataList);

        // Apply filtering if specified
        if ($sharedDataList->filter_field && $sharedDataList->filter_value) {
            $data = $data->where($sharedDataList->filter_field, $sharedDataList->filter_value);
        }

        // Get first 3 records for sample
        $sampleData = $data->take(3)->map(function ($record) use ($columns) {
            $essentialData = [];
            foreach ($columns as $column) {
                $value = $this->getFieldValue($record, $column['value']);
                $essentialData[$column['value']] = $value;
            }

            return $essentialData;
        });

        return [
            'id' => $sharedDataList->id,
            'name' => $sharedDataList->name,
            'base_table' => $sharedDataList->base_table,
            'record_count' => $data->count(),
            'columns' => $columns,
            'data' => $sampleData,
            'generated_at' => now()->toISOString(),
        ];
    }

    /**
     * Get app configuration from pianola/app.json
     */
    protected function getAppConfig()
    {
        if (! File::exists(config_path('pianola/app.json'))) {
            return [
                'colorPrimary' => '#007cba',
                'logoPath' => null,
                'appName' => 'Shared Data Portal',
            ];
        }

        $config = json_decode(File::get(config_path('pianola/app.json')), true);
        $basicConfig = $config['basicConfig'] ?? [];

        return [
            'colorPrimary' => $basicConfig['colorPrimary'] ?? '#007cba',
            'logoPath' => $basicConfig['logoPath'] ?? null,
            'appName' => $basicConfig['appName'] ?? 'Shared Data Portal',
            'appLanguage' => $basicConfig['appLanguage'] ?? 'en',
        ];
    }

    /**
     * Get module configuration for a PianolaList
     */
    protected function getModuleConfig($pianolaList)
    {
        if (! $pianolaList || ! $pianolaList->module) {
            return null;
        }

        if (! File::exists(config_path('pianola/app.json'))) {
            return null;
        }

        $config = json_decode(File::get(config_path('pianola/app.json')), true);
        $modules = $config['modules'] ?? [];

        // Find module by name (case-insensitive match)
        $listModuleName = strtolower($pianolaList->module);

        foreach ($modules as $module) {
            $basicConfig = $module['basicConfig'] ?? [];
            $moduleName = strtolower($basicConfig['name'] ?? '');

            if ($moduleName === $listModuleName) {
                return $basicConfig;
            }
        }

        return null;
    }

    /**
     * Get default sort order for a PianolaList from module configuration
     */
    protected function getDefaultSort($pianolaList)
    {
        $moduleConfig = $this->getModuleConfig($pianolaList);

        return $moduleConfig['defaultSort'] ?? null;
    }

    /**
     * Check if calendar is available for a PianolaList from module configuration
     */
    protected function getHasCalendar($pianolaList)
    {
        $moduleConfig = $this->getModuleConfig($pianolaList);

        return $moduleConfig['hasCalendar'] ?? false;
    }

    /**
     * Get calendar data by mapping fields from schema.json (similar to CalendarPrintoutController)
     */
    protected function getCalendarData($pianolaQuery, $data, $forPrint = false)
    {
        if (! $pianolaQuery || ! $pianolaQuery->base_table) {
            return [];
        }

        // Get calendar mapping from schema.json
        $calendarMapping = $this->getCalendarMapping($pianolaQuery->base_table);
        if (! $calendarMapping) {
            return [];
        }

        // Transform data to standardized calendar format (like CalendarPrintoutController)
        return $this->transformToCalendarEvents($data, $calendarMapping, $forPrint);
    }

    /**
     * Transform data to calendar events format (similar to CalendarPrintoutController)
     */
    protected function transformToCalendarEvents($data, $calendarMapping, $forPrint = false)
    {
        $map = [
            'id' => $calendarMapping['id'] ?? 'id',
            'date' => $calendarMapping['date'] ?? 'date',
            'timestamp_start' => $calendarMapping['timestamp_start'] ?? 'timestamp_start',
            'timestamp_end' => $calendarMapping['timestamp_end'] ?? 'timestamp_end',
            'location' => $calendarMapping['location'] ?? 'location_name',
            'name' => $calendarMapping['name'] ?? 'calendar_event_name',
            'more_information' => $calendarMapping['more_information'] ?? 'calendar_information',
            'locked' => $calendarMapping['locked'] ?? 'calendar_locked',
            'color' => $calendarMapping['color'] ?? 'calendar_color',
        ];

        return collect($data)->map(function ($record) use ($map, $forPrint) {
            // Skip records without required fields
            if (empty(Arr::get($record, $map['date']))) {
                return null;
            }
            if (empty(Arr::get($record, $map['timestamp_start']))) {
                return null;
            }

            // Set default end time if missing (15 minutes after start)
            $timestampEnd = Arr::get($record, $map['timestamp_end']);
            if (empty($timestampEnd)) {
                $startTime = Arr::get($record, $map['timestamp_start']);
                $timestampEnd = date('Y-m-d H:i:s', strtotime($startTime) + 900);
            }

            // Truncate timestamp_end if it goes beyond midnight (for calendar and print views)
            if ($forPrint) {
                $eventDate = Arr::get($record, $map['date']);
                $endDate = date('Y-m-d', strtotime($timestampEnd));

                // If the end timestamp is on a different date than the event date, truncate to 23:59 of the event date
                if ($endDate !== $eventDate) {
                    $timestampEnd = $eventDate . ' 23:59:00';
                }
            }

            return [
                'id' => Arr::get($record, $map['id']),
                'date' => Arr::get($record, $map['date']),
                'calendar_id' => Arr::get($record, $map['id']),
                'calendar_timestamp_start' => Arr::get($record, $map['timestamp_start']),
                'calendar_timestamp_end' => $timestampEnd,
                'calendar_location' => Arr::get($record, $map['location']),
                'calendar_name' => Arr::get($record, $map['name']),
                'calendar_more_information' => Arr::get($record, $map['more_information']),
                'calendar_locked' => Arr::get($record, $map['locked']),
                'calendar_color' => $this->correctColor(Arr::get($record, $map['color'])),
            ];
        })->filter()->sortBy([
            ['date', 'asc'],
            ['calendar_location', 'asc'],
            ['calendar_timestamp_start', 'asc']
        ])->values()->toArray();
    }

    /**
     * Correct color format (from CalendarPrintoutController)
     */
    protected function correctColor($color)
    {
        if (empty($color)) {
            return '#ccc';
        }
        if (strpos($color, '#') === false) {
            return '#'.$color;
        }

        return $color;
    }

    /**
     * Get calendar mapping from schema.json for a specific table
     */
    protected function getCalendarMapping($tableName)
    {
        if (! File::exists(config_path('pianola/schema.json'))) {
            return null;
        }

        $schemas = json_decode(File::get(config_path('pianola/schema.json')), true);

        if (! is_array($schemas)) {
            return null;
        }

        // Find schema by table name
        foreach ($schemas as $schema) {
            if (isset($schema['name']) && $schema['name'] === $tableName) {
                return $schema['appCalendarMapping'] ?? null;
            }
        }

        return null;
    }

    /**
     * Filter data collection to only include columns that will be displayed
     * This reduces response size by removing unused database fields
     */
    protected function filterDataByColumns($data, $columns)
    {
        if (empty($columns)) {
            return $data;
        }

        // Extract column field names
        $columnFields = collect($columns)->pluck('value')->filter()->toArray();

        // Add 'id' field if not already included (needed for Vue key binding)
        if (! in_array('id', $columnFields)) {
            $columnFields[] = 'id';
        }

        // Filter each record to only include needed fields
        return $data->map(function ($record) use ($columnFields) {
            $filteredRecord = [];

            foreach ($columnFields as $field) {
                // Handle nested field access (e.g., 'user.name')
                if (strpos($field, '.') !== false) {
                    $value = $this->getNestedValue($record, $field);
                } else {
                    // Since QueryBuilder returns Eloquent models, use object property access
                    $value = $record->$field ?? null;
                }

                $filteredRecord[$field] = $value;
            }

            return $filteredRecord;
        });
    }

    /**
     * Get nested value from record (e.g., 'user.name' from record)
     * Handles Eloquent models and their relationships
     */
    protected function getNestedValue($record, $field)
    {
        $keys = explode('.', $field);
        $value = $record;

        foreach ($keys as $key) {
            if (is_object($value)) {
                // For Eloquent models, use property access or relationship loading
                if (isset($value->$key)) {
                    $value = $value->$key;
                } elseif (method_exists($value, $key)) {
                    // Try to load relationship if it's a method
                    try {
                        $value = $value->$key;
                    } catch (\Exception $e) {
                        return null;
                    }
                } else {
                    return null;
                }
            } elseif (is_array($value) && isset($value[$key])) {
                $value = $value[$key];
            } else {
                return null;
            }
        }

        return $value;
    }

    /**
     * Filter events for print based on type and date
     */
    protected function filterEventsForPrint($calendarData, $type, $date)
    {
        if ($type === 'daily') {
            return collect($calendarData)->filter(function ($event) use ($date) {
                return $event['date'] === $date;
            })->values()->toArray();
        }

        // Weekly view - get events for the week containing the date
        $mondayBefore = date('Y-m-d', strtotime($date.' - '.date('w', strtotime($date)) - 1 .' days'));
        $weekEnd = date('Y-m-d', strtotime($mondayBefore.' + 6 days'));

        return collect($calendarData)->filter(function ($event) use ($mondayBefore, $weekEnd) {
            return $event['date'] >= $mondayBefore && $event['date'] <= $weekEnd;
        })->values()->toArray();
    }

    /**
     * Get unique locations from filtered events
     */
    protected function getLocationsForPrint($events)
    {
        return collect($events)
            ->pluck('calendar_location')
            ->filter()
            ->unique()
            ->values()
            ->toArray();
    }

    /**
     * Generate print title based on date and type
     */
    protected function getPrintTitle($date, $type)
    {
        $date = date('d.m.Y', strtotime($date));
        $weekday = match (date('l', strtotime($date))) {
            'Monday' => 'Montag',
            'Tuesday' => 'Dienstag',
            'Wednesday' => 'Mittwoch',
            'Thursday' => 'Donnerstag',
            'Friday' => 'Freitag',
            'Saturday' => 'Samstag',
            'Sunday' => 'Sonntag',
        };

        if ($type === 'daily') {
            return 'Tagesübersicht '.$weekday.', '.$date;
        }

        $mondayBefore = date('Y-m-d', strtotime($date.' - '.date('w', strtotime($date)) - 1 .' days'));

        return 'Wochenübersicht: '.date('d.m.Y', strtotime($mondayBefore)).' - '.date('d.m.Y', strtotime($mondayBefore.' + 6 days'));
    }

    /**
     * Get all periods data for printing (all days or all weeks)
     */
    protected function getAllPeriodsForPrint($calendarData, $type)
    {
        $allPeriods = [];

        if ($type === 'daily') {
            // Get all unique dates
            $dates = collect($calendarData)->pluck('date')->unique()->sort()->values();

            foreach ($dates as $date) {
                $events = collect($calendarData)->filter(function ($event) use ($date) {
                    return $event['date'] === $date;
                })->values()->toArray();

                if (! empty($events)) {
                    $locations = $this->getLocationsForPrint($events);
                    $title = $this->getPrintTitle($date, 'daily');

                    $allPeriods[] = [
                        'date' => $date,
                        'events' => $events,
                        'locations' => $locations,
                        'title' => $title,
                        'type' => 'daily',
                    ];
                }
            }
        } else {
            // Get all unique weeks
            $dates = collect($calendarData)->pluck('date')->unique()->sort()->values();
            $weeks = [];

            foreach ($dates as $date) {
                $mondayBefore = date('Y-m-d', strtotime($date.' - '.date('w', strtotime($date)) - 1 .' days'));
                if (! in_array($mondayBefore, $weeks)) {
                    $weeks[] = $mondayBefore;
                }
            }

            foreach ($weeks as $weekStart) {
                $weekEnd = date('Y-m-d', strtotime($weekStart.' + 6 days'));

                $events = collect($calendarData)->filter(function ($event) use ($weekStart, $weekEnd) {
                    return $event['date'] >= $weekStart && $event['date'] <= $weekEnd;
                })->values()->toArray();

                if (! empty($events)) {
                    $locations = $this->getLocationsForPrint($events);
                    $title = $this->getPrintTitle($weekStart, 'weekly');

                    $allPeriods[] = [
                        'date' => $weekStart,
                        'events' => $events,
                        'locations' => $locations,
                        'title' => $title,
                        'type' => 'weekly',
                    ];
                }
            }
        }

        return $allPeriods;
    }

    /**
     * Check if a user can share data for a specific module based on base_table
     */
    protected function userCanShareDataForModule($user, $baseTable)
    {
        if (!$user) {
            return false;
        }

        // Admin users can always share data
        if ($user->is_admin) {
            return true;
        }

        // Get the module name for this base table
        $moduleName = $this->getModuleNameForBaseTable($baseTable);
        if (!$moduleName) {
            return false;
        }

        // Check if user has specific permission for this module
        $enabledModules = json_decode($user->enabled_data_share_modules ?? '[]', true);

        return in_array($moduleName, $enabledModules);
    }

    /**
     * Get module name for a given base table by looking up app.json
     */
    protected function getModuleNameForBaseTable($baseTable)
    {
        if (!File::exists(config_path('pianola/app.json'))) {
            return null;
        }

        $config = json_decode(File::get(config_path('pianola/app.json')), true);
        $modules = $config['modules'] ?? [];

        // Find module where basicConfig.apiEndpoint matches the base table
        foreach ($modules as $module) {
            $basicConfig = $module['basicConfig'] ?? [];
            $apiEndpoint = $basicConfig['apiEndpoint'] ?? '';

            if ($apiEndpoint === $baseTable) {
                return $basicConfig['name'] ?? null;
            }
        }

        return null;
    }
}
