<?php

namespace BalletMecanique\PianolaLaravel\Services;

use Illuminate\Support\Facades\File;
use Illuminate\Support\Str;

class RelationshipBuilder
{
    protected $endpoint;

    protected $schema;

    public function __construct($endpoint, $recordId)
    {
        $this->endpoint = str_replace('-', '_', $endpoint);
        $this->schema = json_decode(File::get(config_path('/pianola/schema.json')), true);
    }

    protected function getRelatedHasManyTables($table)
    {
        return collect(collect($this->schema)->where('name', $table)->pluck('hasMany')->first())->pluck('name');
    }

    protected function getCustomAttributes($table)
    {
        return collect(collect($this->schema)->where('name', $table)->pluck('custom_attributes')->first());
    }

    protected function getRelatedBelongsToTables($table)
    {
        $table_names = collect(collect($this->schema)->where('name', $table)->pluck('belongsTo')->first())->pluck('name');
        $array = [];
        foreach ($table_names as $table_name) {
            if ($table_name !== $this->endpoint) {
                $array[$table_name] = [];
            }
        }

        return $array;
    }

    protected function generateRelationshipTree()
    {
        $related_has_many_tables = $this->getRelatedHasManyTables($this->endpoint);
        $related_belongs_to_tables = $this->getRelatedBelongsToTables($this->endpoint);
        $tree = [];
        foreach ($related_has_many_tables as $related_table) {
            $related_tables_level_2 = $this->getRelatedBelongsToTables($related_table, $this->endpoint);
            $tree['hasMany'][$related_table] = $related_tables_level_2;
            foreach ($related_tables_level_2 as $related_table_level_2 => $emptyValue) {
                $related_tables_level_3 = $this->getRelatedBelongsToTables($related_table_level_2, $this->endpoint);
                $tree['hasMany'][$related_table][$related_table_level_2] = $related_tables_level_3;
            }
        }
        foreach ($related_belongs_to_tables as $related_table => $emptyValue) {
            $tree['belongsTo'][$related_table] = $this->buildNestedBelongsTo($related_table, [$this->endpoint]);
        }

        return $tree;
    }

    protected function buildNestedBelongsTo($table, $visited = [])
    {
        // Prevent infinite loops by tracking visited tables
        if (in_array($table, $visited)) {
            return [];
        }

        $visited[] = $table;
        $related_tables = $this->getRelatedBelongsToTables($table);
        $nested = [];

        foreach ($related_tables as $related_table => $emptyValue) {
            $nested[$related_table] = $this->buildNestedBelongsTo($related_table, $visited);
        }

        return $nested;
    }

    public function getLoadArray()
    {
        $tree = $this->generateRelationshipTree();

        $string_array = [];
        if (isset($tree['hasMany'])) {
            foreach ($tree['hasMany'] as $key_level_1 => $table_level_1) {
                if ($table_level_1) {
                    foreach ($table_level_1 as $key_level_2 => $table_level_2) {
                        if ($table_level_2) {
                            foreach ($table_level_2 as $key_level_3 => $table_level_3) {
                                $string_array[] = $key_level_1.'.'.Str::singular($key_level_2).'.'.Str::singular($key_level_3);
                            }
                        } else {
                            $string_array[] = $key_level_1.'.'.Str::singular($key_level_2);
                        }
                    }
                } else {
                    $string_array[] = $key_level_1;
                }
            }
        }

        if (isset($tree['belongsTo'])) {
            $belongsToArray = $this->flattenBelongsToTree($tree['belongsTo']);
            array_push($string_array, ...$belongsToArray);
        }

        return $string_array;
    }

    public function getHasManyLoadArray()
    {
        $tree = $this->generateRelationshipTree();

        $string_array = [];
        if (isset($tree['hasMany'])) {
            foreach ($tree['hasMany'] as $key_level_1 => $table_level_1) {
                if ($table_level_1) {
                    foreach ($table_level_1 as $key_level_2 => $table_level_2) {
                        if ($table_level_2) {
                            foreach ($table_level_2 as $key_level_3 => $table_level_3) {
                                $string_array[] = $key_level_1.'.'.Str::singular($key_level_2).'.'.Str::singular($key_level_3);
                            }
                        } else {
                            $string_array[] = $key_level_1.'.'.Str::singular($key_level_2);
                        }
                    }
                } else {
                    $string_array[] = $key_level_1;
                }
            }
        }

        return $string_array;
    }

    public function getBelongsToLoadArray()
    {
        $tree = $this->generateRelationshipTree();
        $string_array = [];
        if (isset($tree['belongsTo'])) {
            $string_array = $this->flattenBelongsToTree($tree['belongsTo']);
        }

        return $string_array;
    }

    protected function flattenBelongsToTree($tree, $prefix = '')
    {
        $result = [];

        foreach ($tree as $table => $nested) {
            $currentPath = $prefix ? $prefix . '.' . Str::singular($table) : Str::singular($table);
            $result[] = $currentPath;

            if (!empty($nested) && is_array($nested)) {
                $nestedResults = $this->flattenBelongsToTree($nested, $currentPath);
                array_push($result, ...$nestedResults);
            }
        }

        return $result;
    }

    public function getExtraLoadArray()
    {
        $string = collect($this->schema)->where('name', $this->endpoint)->pluck('load_extra')->first();
        if ($string) {
            return explode(',', $string);
        }

        return null;
    }

    public function getAppendLoadArray()
    {
        return $this->getCustomAttributes($this->endpoint)->map(function ($customAttribute) {
            return $customAttribute['relationshipDependencies'] ?? [];
        })->filter()->flatten()->unique()->toArray();
    }

    public function getOmittedFromExcelBelongsToTables()
    {
        $tables = collect(collect($this->schema)->where('name', $this->endpoint)->pluck('belongsTo')->first());

        return $tables->map(function ($table) {
            if ($table['omit_from_load_excel'] ?? null) {
                return Str::singular($table['name']);
            }

            return null;
        })->filter()->values();
    }
}
