<?php

namespace BalletMecanique\PianolaLaravel\Http\Controllers;

use BalletMecanique\PianolaLaravel\Models\PianolaCalendarPreset;
use BalletMecanique\PianolaLaravel\Services\QueryBuilder;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Str;

class CalendarPrintoutController
{
    public function __invoke($endpoint, Request $request)
    {
        $type = $request->input('type');
        $date = $request->input('date');
        $events = $this->getEvents($endpoint, $request, $type, $date);
        $locations = $this->getLocations($endpoint, $request);
        $logoPath = $this->getLogoPath();
        $title = $this->getTitle($date, $type);

        return view('pianola-laravel::printouts.calendar', compact('events', 'date', 'locations', 'title', 'logoPath', 'type'));
    }

    protected function getEvents($endpoint, $request, $type, $date)
    {
        $modelName = 'App\\Models\\'.Str::singular(Str::studly($endpoint));
        $model = new $modelName();
        $query = new QueryBuilder($model, $request);
        $query = $query->baseQuery();
        if ($type === 'daily') {
            $query->where('date', $date);
        } else {
            $mondayBefore = date('Y-m-d', strtotime($date.' - '.date('w', strtotime($date)) - 1 .' days'));
            $query->where('date', '>=', date('Y-m-d', strtotime($mondayBefore)))
                ->where('date', '<=', date('Y-m-d', strtotime($mondayBefore.' + 6 days')))
                ->whereIn('location_id', $this->getLocationIds($endpoint, $request));
        }
        $records = $query->get();
        $records = $this->returnOnlyCalendarData($records, $endpoint);

        return $records->values();
    }

    protected function getLocationIds($endpoint, $request)
    {
        $preset = PianolaCalendarPreset::find($request->input('preset'));
        $presetData = json_decode($preset->json, false, 512, JSON_THROW_ON_ERROR);

        return collect($presetData)->where('number', $request->input('calPage'))->pluck('location_ids')->first();
    }

    protected function returnOnlyCalendarData($records, $endpoint)
    {
        $schema = collect(json_decode(File::get(config_path('/pianola/schema.json')), true));
        $endpointSchema = $schema->where('name', $endpoint)->first();
        $calendarMapping = $endpointSchema['appCalendarMapping'] ?? [];
        $map = [
            'id' => $calendarMapping['id'] ?? 'id',
            'date' => $calendarMapping['date'] ?? 'date',
            'timestamp_start' => $calendarMapping['timestamp_start'] ?? 'timestamp_start',
            'timestamp_end' => $calendarMapping['timestamp_end'] ?? 'timestamp_end',
            'location' => $calendarMapping['location'] ?? 'location_name',
            'name' => $calendarMapping['name'] ?? 'calendar_event_name',
            'more_information' => $calendarMapping['more_information'] ?? 'calendar_information',
            'locked' => $calendarMapping['locked'] ?? 'calendar_locked',
            'color' => $calendarMapping['color'] ?? 'calendar_color',
        ];
        $customAttributes = $endpointSchema['custom_attributes'] ?? [];
        $relationshipDependencies = collect($customAttributes)->filter(function ($attribute) use ($map) {
            return in_array($attribute['name'], $map);
        })->pluck('relationshipDependencies')->flatten();

        $records->load($relationshipDependencies->toArray());

        return $records->map(function ($record) use ($map) {
            if (empty($record[$map['date']])) {
                return null;
            }
            if (empty($record[$map['timestamp_start']])) {
                return null;
            }
            if (empty($record[$map['timestamp_end']])) {
                $record[$map['timestamp_end']] = date('Y-m-d H:i:s', strtotime($record[$map['timestamp_start']]) + 900);
            }

            return [
                'date' => $record[$map['date']],
                'calendar_id' => $record[$map['id']],
                'calendar_timestamp_start' => $record[$map['timestamp_start']],
                'calendar_timestamp_end' => $record[$map['timestamp_end']],
                'calendar_location' => $record[$map['location']],
                'calendar_name' => $record[$map['name']],
                'calendar_more_information' => $record[$map['more_information']],
                'calendar_locked' => $record[$map['locked']],
                'calendar_color' => $this->correctColor($record[$map['color']]),
            ];
        })->filter()->values();
    }

    protected function correctColor($color)
    {
        if (empty($color)) {
            return '#ccc';
        }
        if (strpos($color, '#') === false) {
            return '#'.$color;
        }

        return $color;
    }

    protected function getLocations($endpoint, $request)
    {
        $preset = PianolaCalendarPreset::find($request->input('preset'));
        $presetData = json_decode($preset->json, false, 512, JSON_THROW_ON_ERROR);
        $locationIds = collect($presetData)->where('number', $request->input('calPage'))->pluck('location_ids')->first();
        $appJson = file_get_contents(config_path('pianola/app.json'));
        $app = json_decode($appJson, true, 512, JSON_THROW_ON_ERROR);

        //find $app->modules where ->basicConfig->moduleEndpoint === $endpoint
        $module = collect($app['modules'])->where('basicConfig.apiEndpoint', $endpoint)->first();
        $calendarLocationNameColumn = $module['basicConfig']['calendarLocationNameColumn'] ?? 'name';

        return collect($locationIds)->map(function ($locationId) use ($calendarLocationNameColumn) {
            $modelName = 'App\\Models\\Location';
            $model = new $modelName();
            return $model::find($locationId)->{$calendarLocationNameColumn};
        })->values();
    }

    protected function getLogoPath()
    {
        $appJson = file_get_contents(config_path('pianola/app.json'));
        $app = json_decode($appJson, true, 512, JSON_THROW_ON_ERROR);

        return $app['basicConfig']['logoPath'];
    }

    protected function getTitle($date, $type)
    {
        $date = date('d.m.Y', strtotime($date));
        $weekday = match (date('l', strtotime($date))) {
            'Monday' => 'Montag',
            'Tuesday' => 'Dienstag',
            'Wednesday' => 'Mittwoch',
            'Thursday' => 'Donnerstag',
            'Friday' => 'Freitag',
            'Saturday' => 'Samstag',
            'Sunday' => 'Sonntag',
        };
        if ($type === 'daily') {
            return 'Tagesübersicht '.$weekday.', '.$date;
        }
        $mondayBefore = date('Y-m-d', strtotime($date.' - '.date('w', strtotime($date)) - 1 .' days'));

        return 'Wochenübersicht: '.date('d.m.Y', strtotime($mondayBefore)).' - '.date('d.m.Y', strtotime($mondayBefore.' + 6 days'));
    }
}
