<?php

namespace BalletMecanique\PianolaLaravel\Commands;

use BalletMecanique\PianolaLaravel\Models\User;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Hash;

class PianolaInstallCommand extends Command
{
    public $signature = 'pianola:install';

    public $description = 'Installs Pianola';

    public function handle(): int
    {
        //create app config files, if none present
        $this->createDirectory(config_path('/pianola'));
        $this->createFile(config_path('/pianola/app.json'), $this->getAppJson());
        $this->createFile(config_path('/pianola/schema.json'), $this->getSchemaJson());

        //create migrations for starter app
        Artisan::call('vendor:publish --tag="pianola-laravel-migrations"');
        Artisan::call('migrate');

        //create models and factories for starter app
        $this->createFile(app_path('/Models/Person.php'), $this->getPersonModelString());
        $this->createFile(database_path('/factories/PersonFactory.php'), $this->getPersonModelFactoryString());
        $this->createFile(app_path('/Models/Company.php'), $this->getCompanyModelString());
        $this->createFile(database_path('/factories/CompanyFactory.php'), $this->getCompanyModelFactoryString());
        $this->createFile(app_path('/Models/Employee.php'), $this->getEmployeeModelString());
        $this->createFile(database_path('/factories/EmployeeFactory.php'), $this->getEmployeeModelFactoryString());
        $this->createFile(app_path('/Models/Country.php'), $this->getCountryModelString());

        //seed the test database
        $this->createFile(database_path('/seeders/TemplateAppSeeder.php'), $this->getTemplateAppSeederString());
        Artisan::call('db:seed --class=TemplateAppSeeder');

        //create dev user
        $this->createDevUser();

        return self::SUCCESS;
    }

    protected function createFile($filepath, $content)
    {
        if (! file_exists($filepath)) {
            File::put($filepath, $content);
        }
    }

    protected function createDirectory($filepath)
    {
        if (! file_exists($filepath)) {
            File::makeDirectory($filepath);
        }
    }

    protected function createDevUser()
    {
        if (User::first()) {
            return;
        }
        //create new user with feedback
        $this->info('Create a local dev user for the app:');
        $name = $this->ask('What is your name?');
        while ($name == '') {
            $name = $this->ask('Name cannot be empty. What is your name?');
        }
        $email = $this->ask('What is your mail address?');
        while ($email == '') {
            $email = $this->ask('Email address cannot be empty. What is your mail address?');
        }
        $password = $this->ask('What initial password do you want to use?', 'password');

        User::create([
            'name' => $name,
            'email' => $email,
            'password' => Hash::make($password),
            'is_dev' => 1,
            'is_admin' => 1,
            'is_active' => 1,
            'modules' => json_encode([]),
        ]);
    }

    protected function getAppJson()
    {
        return <<<'EOT'
        {"modules":[{"tabs":[{"name":"Basic Data","components":[{"props":{"components":[{"props":{"components":[{"props":{"field":"first_name","label":"First Name","isReadonly":false},"componentType":"BaseInputText"},{"props":{"field":"last_name","label":"Last Name"},"componentType":"BaseInputText"},{"props":{"field":"street_1","label":"Street 1","isReadonly":false},"componentType":"BaseInputText"},{"props":{"field":"street_2","label":"Street 2"},"componentType":"BaseInputText"},{"props":{"components":[{"props":{"columns":"4","components":[{"props":{"field":"post_code","label":"Post Code"},"componentType":"BaseInputText"}]},"componentType":"BaseLayoutColumn"},{"props":{"columns":"8","components":[{"props":{"field":"city","label":"City"},"componentType":"BaseInputText"}]},"componentType":"BaseLayoutColumn"}]},"componentType":"BaseLayoutRow"},{"props":{"field":"country_id","label":"Country","endpoint":"countries","itemText":"name_en","itemValue":"id"},"componentType":"BaseInputAutocomplete"}],"borderRight":false},"componentType":"BaseLayoutColumn"},{"props":{"components":[{"props":{"field":"phone","label":"Phone","buttonType":"enlarge"},"componentType":"BaseInputText"},{"props":{"field":"mobile","label":"Mobile","buttonType":"enlarge"},"componentType":"BaseInputText"},{"props":{"field":"email","label":"E-Mail","buttonType":"mail"},"componentType":"BaseInputText"},{"props":{"field":"website","label":"Website","buttonType":"url"},"componentType":"BaseInputText"}]},"componentType":"BaseLayoutColumn"}]},"componentType":"BaseLayoutRow"},{"props":{"marginTop":"3","marginBottom":"3"},"componentType":"BaseDivider"},{"props":{"components":[{"props":{"components":[{"props":{"title":"Business Address(es)","endpoint":"employees","linkPath":"companies","components":[{"props":{"field":"company_id","label":"Company","headers":[{"text":"Name","value":"name"},{"text":"City","value":"city"}],"endpoint":"companies","components":[{"props":{"field":"name","label":"Name"},"componentType":"BaseInputText"}],"displayField":"name","displayValue":"name","newRecordLabel":"new company"},"componentType":"BaseInputSearchOrCreate"},{"props":{"field":"function","label":"Function"},"componentType":"BaseInputText"},{"props":{"field":"department","label":"Department"},"componentType":"BaseInputText"},{"props":{"field":"business_phone","label":"Business Phone"},"componentType":"BaseInputText"},{"props":{"field":"business_mobile","label":"Business Mobile"},"componentType":"BaseInputText"},{"props":{"field":"business_email","label":"Business Email"},"componentType":"BaseInputText"}],"noNewRecord":false,"maxDialogWidth":450,"newSubrecordButtonText":"new business address"},"componentType":"BaseCards"}]},"componentType":"BaseLayoutColumn"},{"props":{"components":[]},"componentType":"BaseLayoutColumn"}]},"componentType":"BaseLayoutRow"}]}],"lists":[{"name":"Default","headers":[{"text":"First Name","value":"first_name"},{"text":"Last Name","value":"last_name"},{"text":"Street 1","value":"street_1"},{"text":"Street 2","value":"street_2"},{"text":"Post Code","value":"post_code"},{"text":"City","value":"city"},{"text":"Country","value":"country.name_en"}]}],"basicConfig":{"icon":"mdi-account-group","name":"Contacts","path":"contacts","divider":false,"iconRecord":"mdi-account","apiEndpoint":"people","defaultSort":"last_name,first_name","moduleColor":null,"newRecordText":"New Contact","loadRelatedData":true},"recordTitleConfig":{"fieldTitle":"full_name"},"newRecordModalComponents":[{"props":{"field":"first_name","label":"First Name"},"componentType":"BaseInputText"},{"props":{"field":"last_name","label":"Last Name"},"componentType":"BaseInputText"}]},{"tabs":[{"name":"Basic Data","components":[{"props":{"components":[{"props":{"components":[{"props":{"field":"name","label":"Name","isReadonly":false},"componentType":"BaseInputText"},{"props":{"field":"street_1","label":"Street 1"},"componentType":"BaseInputText"},{"props":{"field":"street_2","label":"Street 2"},"componentType":"BaseInputText"},{"props":{"components":[{"props":{"columns":"4","components":[{"props":{"field":"post_code","label":"Post Code"},"componentType":"BaseInputText"}]},"componentType":"BaseLayoutColumn"},{"props":{"columns":"8","components":[{"props":{"field":"city","label":"City"},"componentType":"BaseInputText"}]},"componentType":"BaseLayoutColumn"}]},"componentType":"BaseLayoutRow"},{"props":{"field":"country_id","label":"Country","endpoint":"countries","itemText":"name_en","itemValue":"id"},"componentType":"BaseInputAutocomplete"}],"borderRight":false},"componentType":"BaseLayoutColumn"},{"props":{"components":[{"props":{"field":"phone","label":"Phone","buttonType":"enlarge"},"componentType":"BaseInputText"},{"props":{"field":"email","label":"E-Mail","buttonType":"mail"},"componentType":"BaseInputText"},{"props":{"field":"website","label":"Website","buttonType":"url"},"componentType":"BaseInputText"}]},"componentType":"BaseLayoutColumn"}]},"componentType":"BaseLayoutRow"},{"props":{"components":[{"props":{"maxWidth":1200,"components":[{"props":{"title":"Employees","headers":[{"text":"Name","value":"person.full_name","isLink":true},{"text":"Department","value":"department"},{"text":"Function","value":"function"},{"text":"Phone","value":"business_phone"},{"text":"Mobile","value":"business_mobile"},{"text":"Email","value":"business_email"}],"endpoint":"employees","noDelete":false,"components":[{"props":{"field":"person_id","label":"Contact","headers":[{"text":"Name","value":"full_name"},{"text":"City","value":"city"}],"endpoint":"people","components":[{"props":{"field":"first_name","label":"First Name"},"componentType":"BaseInputText"},{"props":{"field":"last_name","label":"Last Name"},"componentType":"BaseInputText"}],"displayField":"full_name","displayValue":"full_name","newRecordLabel":"new contact"},"componentType":"BaseInputSearchOrCreate"},{"props":{"field":"function","label":"Function"},"componentType":"BaseInputText"},{"props":{"field":"department","label":"Department"},"componentType":"BaseInputText"},{"props":{"field":"business_phone","label":"Business Phone"},"componentType":"BaseInputText"},{"props":{"field":"business_mobile","label":"Business Mobile"},"componentType":"BaseInputText"},{"props":{"field":"business_email","label":"Business Email"},"componentType":"BaseInputText"}],"maxDialogWidth":450,"newSubrecordButtonText":"new employee"},"componentType":"BaseRecordDataTable"}]},"componentType":"BaseLayoutColumn"}]},"componentType":"BaseLayoutRow"}]}],"lists":[{"name":"Default","headers":[{"text":"Name","value":"name"},{"text":"Street 1","value":"street_1"},{"text":"Street 2","value":"street_2"},{"text":"Post Code","value":"post_code"},{"text":"City","value":"city"},{"text":"Country","value":"country.name_en"}]}],"basicConfig":{"icon":"mdi-office-building","name":"Companies","path":"companies","divider":false,"iconRecord":null,"apiEndpoint":"companies","defaultSort":"name","moduleColor":null,"newRecordText":"New Company","loadRelatedData":true},"queryFields":[{"fields":[{"text":"Name","value":"name"},{"text":"Street 1","value":"street_1"},{"text":"Street 2","value":"street_2"},{"text":"Post Code","value":"post_code"},{"text":"City","value":"city"},{"text":"Country","type":"autocomplete","value":"country_id","endpoint":"countries","itemText":"name_en","itemValue":"id"}],"tableName":"companies","tableNameLabel":"Basic Details"}],"recordTitleConfig":{"fieldTitle":"name"},"newRecordModalComponents":[{"props":{"field":"name","label":"Name"},"componentType":"BaseInputText"}]}],"settings":[{"name":"Länder","sort":"name_de+asc","endpoint":"countries","components":[{"props":{"field":"name_de","label":"Name DE"},"componentType":"BaseInputText"},{"props":{"field":"name_en","label":"Name EN"},"componentType":"BaseInputText"},{"props":{"field":"name_fr","label":"Name FR"},"componentType":"BaseInputText"},{"props":{"field":"name_it","label":"Name IT"},"componentType":"BaseInputText"}]}],"basicConfig":{"appName":"Starter Template","logoPath":"https://cdn.jsdelivr.net/gh/ballet-mecanique/pianola-frontend@latest/dist/assets/logo-standalone-black-background-rgb.png","twoFactor":false,"appLanguage":"en","faviconPath":"https://cdn.jsdelivr.net/gh/ballet-mecanique/pianola-frontend@latest/dist/assets/logo-standalone-black-background-rgb.png","logoPadding":1,"colorPrimary":"#45659b","drawerIsDark":false,"apiUrlStaging":null,"exporterIsActive":true,"backgroundPath":"https://cdn.jsdelivr.net/gh/ballet-mecanique/pianola-frontend@latest/dist/assets/background.jpg","colorBackground":"#f2f2ef","colorItemDrawer":"#45659b","apiUrlProduction":"api/pianola/","recordUpdateMode":"auto","apiUrlDevelopment":"http://localhost:8000/pianola/","defaultErrorMessage":"There was a problem connecting to the server. Please try again with a refresh, and if it persists, please contact Andrew on andrew@balletmecanique.eu.","colorBackgroundDrawer":"#dedede","colorBackgroundDrawer2":"#e2b7b9","onlyAdminCanSeeSettings":true}}
        EOT;
    }

    protected function getSchemaJson()
    {
        return <<<'EOT'
        [{"name":"countries","append":null,"columns":[{"id":1456,"name":"id","app_label":null,"column_type":"bigint(20) unsigned","is_nullable":null,"alternate_name":null},{"id":1457,"name":"name_de","app_label":"Name DE","column_type":"varchar(255)","is_nullable":null,"alternate_name":null},{"id":1458,"name":"name_en","app_label":"Name EN","column_type":"varchar(255)","is_nullable":null,"alternate_name":null},{"id":1459,"name":"iso_3166_alpha2","app_label":"ISO 3166 alpha2","column_type":"varchar(255)","is_nullable":null,"alternate_name":null},{"id":1460,"name":"created_at","app_label":null,"column_type":"timestamp","is_nullable":null,"alternate_name":null},{"id":1461,"name":"updated_at","app_label":null,"column_type":"timestamp","is_nullable":null,"alternate_name":null}],"hasMany":[{"name":"people","app_tab_order":null},{"name":"companies","app_tab_order":null}],"belongsTo":[],"load_extra":null,"app_is_module":1,"alternate_name":null,"app_module_icon":null,"app_name_plural":"Länder","app_module_color":null,"app_module_order":null,"app_name_singular":"Land","custom_attributes":[],"app_module_path_name":null,"app_record_filter_columns":null,"app_include_in_global_search":null,"app_record_description_column":null},{"name":"employees","append":null,"columns":[{"id":1497,"name":"id","app_label":null,"column_type":"bigint(20) unsigned","is_nullable":null,"alternate_name":null},{"id":1498,"name":"person_id","app_label":null,"column_type":"bigint(20) unsigned","is_nullable":null,"alternate_name":null},{"id":1499,"name":"company_id","app_label":null,"column_type":"bigint(20) unsigned","is_nullable":null,"alternate_name":null},{"id":1501,"name":"function","app_label":"Function","column_type":"varchar(255)","is_nullable":null,"alternate_name":null},{"id":1502,"name":"department","app_label":"Department","column_type":"varchar(255)","is_nullable":null,"alternate_name":null},{"id":1503,"name":"email","app_label":"E-mail","column_type":"varchar(255)","is_nullable":null,"alternate_name":null},{"id":1504,"name":"phone","app_label":"Phone","column_type":"varchar(255)","is_nullable":null,"alternate_name":null},{"id":1505,"name":"mobile","app_label":"Mobile","column_type":"varchar(255)","is_nullable":null,"alternate_name":null},{"id":1507,"name":"created_at","app_label":null,"column_type":"timestamp","is_nullable":null,"alternate_name":null},{"id":1508,"name":"updated_at","app_label":null,"column_type":"timestamp","is_nullable":null,"alternate_name":null}],"hasMany":[],"belongsTo":[{"name":"people"},{"name":"companies"}],"load_extra":null,"app_is_module":null,"alternate_name":null,"app_module_icon":null,"app_name_plural":null,"app_module_color":null,"app_module_order":null,"app_name_singular":null,"custom_attributes":[],"app_module_path_name":null,"app_record_filter_columns":null,"app_include_in_global_search":null,"app_record_description_column":null},{"name":"companies","append":null,"columns":[{"id":1778,"name":"id","app_label":null,"column_type":"bigint(20) unsigned","is_nullable":null,"alternate_name":null},{"id":1779,"name":"name","app_label":"Name","column_type":"varchar(255)","is_nullable":null,"alternate_name":null},{"id":1781,"name":"street_1","app_label":"Street 1","column_type":"varchar(255)","is_nullable":null,"alternate_name":null},{"id":1782,"name":"street_2","app_label":"Street 2","column_type":"varchar(255)","is_nullable":null,"alternate_name":null},{"id":1783,"name":"post_code","app_label":"Post Code","column_type":"varchar(255)","is_nullable":null,"alternate_name":null},{"id":1784,"name":"city","app_label":"City","column_type":"varchar(255)","is_nullable":null,"alternate_name":null},{"id":1785,"name":"country_id","app_label":null,"column_type":"bigint(20) unsigned","is_nullable":null,"alternate_name":null},{"id":1786,"name":"email","app_label":"E-mail","column_type":"varchar(255)","is_nullable":null,"alternate_name":null},{"id":1787,"name":"website","app_label":"Website","column_type":"varchar(255)","is_nullable":null,"alternate_name":null},{"id":1788,"name":"phone","app_label":"Phone","column_type":"varchar(255)","is_nullable":null,"alternate_name":null},{"id":1797,"name":"created_at","app_label":null,"column_type":"timestamp","is_nullable":null,"alternate_name":null},{"id":1798,"name":"updated_at","app_label":null,"column_type":"timestamp","is_nullable":null,"alternate_name":null}],"hasMany":[{"name":"employees","app_tab_order":null}],"belongsTo":[{"name":"countries"}],"app_is_module":1,"alternate_name":null,"app_module_icon":"mdi-office-building","app_name_plural":null,"app_module_color":null,"app_module_order":null,"app_name_singular":null,"custom_attributes":[],"app_module_path_name":"companies","app_record_filter_columns":"name","app_include_in_global_search":1,"app_record_description_column":"name"},{"name":"people","append":null,"columns":[{"id":1805,"name":"id","app_label":null,"column_type":"bigint(20) unsigned","is_nullable":null,"alternate_name":null},{"id":1806,"name":"last_name","app_label":"Last Name","column_type":"varchar(255)","is_nullable":null,"alternate_name":null},{"id":1807,"name":"first_name","app_label":"First Name","column_type":"varchar(255)","is_nullable":null,"alternate_name":null},{"id":1812,"name":"street_1","app_label":"Street 1","column_type":"varchar(255)","is_nullable":null,"alternate_name":null},{"id":1813,"name":"street_2","app_label":"Street 2","column_type":"varchar(255)","is_nullable":null,"alternate_name":null},{"id":1814,"name":"post_code","app_label":"Post Code","column_type":"varchar(255)","is_nullable":null,"alternate_name":null},{"id":1815,"name":"city","app_label":"City","column_type":"varchar(255)","is_nullable":null,"alternate_name":null},{"id":1816,"name":"country_id","app_label":"Country","column_type":"bigint(20) unsigned","is_nullable":null,"alternate_name":null},{"id":1817,"name":"email","app_label":"Email","column_type":"varchar(255)","is_nullable":null,"alternate_name":null},{"id":1833,"name":"phone","app_label":"Phone","column_type":"varchar(255)","is_nullable":null,"alternate_name":null},{"id":1835,"name":"mobile","app_label":"Mobile","column_type":"varchar(255)","is_nullable":null,"alternate_name":null},{"id":1836,"name":"website","app_label":"Website","column_type":"varchar(255)","is_nullable":null,"alternate_name":null},{"id":1841,"name":"created_at","app_label":"created at","column_type":"timestamp","is_nullable":null,"alternate_name":null},{"id":1842,"name":"updated_at","app_label":"updated at","column_type":"timestamp","is_nullable":null,"alternate_name":null}],"hasMany":[{"name":"employees","app_tab_order":null}],"belongsTo":[{"name":"countries"}],"load_extra":null,"app_is_module":1,"alternate_name":null,"app_module_icon":"mdi-account","app_name_plural":"People","app_module_color":null,"app_module_order":null,"app_name_singular":"Person","custom_attributes":[],"app_module_path_name":"people","app_record_filter_columns":"first_name,last_name","app_include_in_global_search":1,"app_record_description_column":"full_name"}]
        EOT;
    }

    protected function getPersonModelString()
    {
        return <<<'EOT'
        <?php

        namespace App\Models;

        use Illuminate\Database\Eloquent\Factories\HasFactory;
        use Illuminate\Database\Eloquent\Model;

        class Person extends Model
        {
            use HasFactory;
            protected $guarded = [];
            protected $appends = ['full_name'];

            public function companies()
            {
                return $this->belongsToMany('App\Models\Company', 'App\Models\Employee');
            }

            public function employees()
            {
                return $this->hasMany('App\Models\Employee');
            }

            public function country()
            {
                return $this->belongsTo('App\Models\Country');
            }

            public function getFullNameAttribute()
            {
                return $this->first_name . ' ' . $this->last_name;
            }
        }
        EOT;
    }

    protected function getPersonModelFactoryString()
    {
        return <<<'EOT'
        <?php

        namespace Database\Factories;

        use Illuminate\Database\Eloquent\Factories\Factory;

        /**
         * @extends \Illuminate\Database\Eloquent\Factories\Factory<\App\Models\Person>
         */
        class PersonFactory extends Factory
        {
            /**
             * Define the model's default state.
             *
             * @return array<string, mixed>
             */
            public function definition()
            {
                return [
                    'first_name' => $this->faker->firstName(),
                    'last_name' => $this->faker->lastName(),
                    'street_1' => $this->faker->streetName(),
                    'post_code' => $this->faker->postcode(),
                    'city' => $this->faker->city(),
                    'country_id' => $this->faker->numberBetween(0, 193),
                    'phone' => $this->faker->phoneNumber(),
                    'mobile' => $this->faker->phoneNumber(),
                    'email' => $this->faker->unique()->safeEmail(),
                ];
            }
        }
        EOT;
    }

    protected function getCompanyModelFactoryString()
    {
        return <<<'EOT'
        <?php

        namespace Database\Factories;

        use Illuminate\Database\Eloquent\Factories\Factory;
        use Illuminate\Support\Str;

        /**
         * @extends \Illuminate\Database\Eloquent\Factories\Factory<\App\Models\Company>
         */
        class CompanyFactory extends Factory
        {
            /**
             * Define the model's default state.
             *
             * @return array<string, mixed>
             */
            public function definition()
            {
                $name = $this->faker->company();
                $nameDomain = Str::slug($name) . '.' . $this->faker->tld();
                return [
                    'name' => $name,
                    'street_1' => $this->faker->streetName(),
                    'post_code' => $this->faker->postcode(),
                    'city' => $this->faker->city(),
                    'country_id' => $this->faker->numberBetween(0, 193),
                    'phone' => $this->faker->phoneNumber(),
                    'email' => 'info@' . $nameDomain,
                    'website' => $nameDomain
                ];
            }
        }
        EOT;
    }

    protected function getCompanyModelString()
    {
        return <<<'EOT'
        <?php

        namespace App\Models;

        use Illuminate\Database\Eloquent\Factories\HasFactory;
        use Illuminate\Database\Eloquent\Model;

        class Company extends Model
        {
            use HasFactory;
            protected $guarded = [];

            public function employees()
            {
                return $this->hasMany('App\Models\Employee');
            }

            public function country()
            {
                return $this->belongsTo('App\Models\Country');
            }
        }
        EOT;
    }

    protected function getEmployeeModelString()
    {
        return <<<'EOT'
        <?php

        namespace App\Models;

        use Illuminate\Database\Eloquent\Factories\HasFactory;
        use Illuminate\Database\Eloquent\Model;

        class Employee extends Model
        {
            use HasFactory;
            protected $guarded = [];

            public function person()
            {
                return $this->belongsTo('App\Models\Person');
            }

            public function company()
            {
                return $this->belongsTo('App\Models\Company');
            }
        }
        EOT;
    }

    protected function getEmployeeModelFactoryString()
    {
        return <<<'EOT'
        <?php

        namespace Database\Factories;

        use Illuminate\Database\Eloquent\Factories\Factory;

        /**
         * @extends \Illuminate\Database\Eloquent\Factories\Factory<\App\Models\Employee>
         */
        class EmployeeFactory extends Factory
        {
            /**
             * Define the model's default state.
             *
             * @return array<string, mixed>
             */
            public function definition()
            {
                return [
                    'function' => $this->faker->jobTitle(),
                    'department' => $this->faker->word(),
                    'business_phone' => $this->faker->phoneNumber(),
                    'business_mobile' => $this->faker->phoneNumber(),
                    'business_email' => $this->faker->unique()->safeEmail(),
                ];
            }
        }
        EOT;
    }

    protected function getCountryModelString()
    {
        return <<<'EOT'
        <?php

        namespace App\Models;

        use Illuminate\Database\Eloquent\Factories\HasFactory;
        use Illuminate\Database\Eloquent\Model;

        class Country extends Model
        {
            use HasFactory;
            protected $guards = [];

            public function people()
            {
                return $this->hasMany('App\Models\Person');
            }

            public function companies()
            {
                return $this->hasMany('App\Models\Company');
            }
        }
        EOT;
    }

    protected function getTemplateAppSeederString()
    {
        return <<<'EOT'
        <?php

        namespace Database\Seeders;

        use App\Models\Company;
        use App\Models\Person;
        use App\Models\Employee;
        use Illuminate\Database\Console\Seeds\WithoutModelEvents;
        use Illuminate\Database\Seeder;

        class TemplateAppSeeder extends Seeder
        {
            /**
             * Run the database seeds.
             *
             * @return void
             */
            public function run()
            {
                $i = 1;
                while ($i <= 10) {
                    Employee::factory()
                        ->for(Person::factory())
                        ->for(Company::factory())
                        ->create();
                    $i++;
                }
            }
        }
        EOT;
    }
}
