<?php

namespace BalletMecanique\PianolaLaravel\Http\Controllers;

use BalletMecanique\PianolaLaravel\Models\SharedDataList;
use BalletMecanique\PianolaLaravel\Models\SharedDataListMember;
use BalletMecanique\PianolaLaravel\Services\QueryBuilder;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Str;

class SharedDataController
{
    /**
     * Display a listing of shared data
     */
    public function index(Request $request)
    {
        $user = Auth::guard('shared_data')->user();

        // Determine which lists to show based on user type
        if ($user->email === 'test@example.com') {
            // Test user sees all lists (excluding expired ones)
            $lists = SharedDataList::with(['members', 'pianolaList', 'pianolaQuery'])
                ->where(function ($query) {
                    $query->whereNull('expiration_date')
                        ->orWhere('expiration_date', '>=', now()->toDateString());
                })
                ->get();
        } else {
            // Regular users see only lists they're members of (excluding expired ones)
            $listIds = SharedDataListMember::where('email', $user->email)
                ->pluck('pianola_shared_data_list_id');

            $lists = SharedDataList::with(['members', 'pianolaList', 'pianolaQuery'])
                ->whereIn('id', $listIds)
                ->where(function ($query) {
                    $query->whereNull('expiration_date')
                        ->orWhere('expiration_date', '>=', now()->toDateString());
                })
                ->get();
        }

        // Check if this is an API request
        if ($request->expectsJson() || $request->is('api/*')) {
            // Return JSON for API requests
            return response()->json([
                'message' => 'Shared data index',
                'user' => $user,
                'lists' => $lists,
            ]);
        }

        // Get app configuration
        $appConfig = $this->getAppConfig();

        // Set app locale
        app()->setLocale($appConfig['appLanguage']);

        // Return view for web requests
        return view('pianola-laravel::shared-data.index', compact('lists', 'user', 'appConfig'));
    }

    /**
     * Display the specified shared data item
     */
    public function show(Request $request, $id)
    {
        $user = Auth::guard('shared_data')->user();

        // Find the shared data list
        $sharedDataList = SharedDataList::with(['pianolaList', 'pianolaQuery'])->findOrFail($id);

        // Check if the list has expired
        if ($sharedDataList->expiration_date && $sharedDataList->expiration_date < now()->toDateString()) {
            abort(404, 'This shared data list has expired.');
        }

        // Check access permissions (same logic as index)
        if ($user->email !== 'test@example.com') {
            $hasAccess = SharedDataListMember::where('pianola_shared_data_list_id', $id)
                ->where('email', $user->email)
                ->exists();

            if (! $hasAccess) {
                abort(403, 'You do not have access to this shared data list.');
            }
        }

        // Get the data using QueryBuilder
        $data = [];
        $columns = [];

        if ($sharedDataList->pianolaQuery && $sharedDataList->base_table) {
            // Create the model instance
            $modelName = 'App\\Models\\'.Str::singular(Str::studly($sharedDataList->base_table));

            if (class_exists($modelName)) {
                $model = new $modelName;

                // Prepare request parameters for QueryBuilder
                $queryRequest = [
                    'q' => $sharedDataList->pianolaQuery->query,
                    'page' => 'all', // Get all results for shared data
                ];

                // Add constant pre-filter if filter_field and filter_value are set
                if ($sharedDataList->filter_field && $sharedDataList->filter_value) {
                    $queryRequest['c'] = $sharedDataList->filter_field.'|'.$sharedDataList->filter_value;
                }

                // Add default sort order if available
                $defaultSort = $this->getDefaultSort($sharedDataList->pianolaList);
                if ($defaultSort) {
                    $queryRequest['sort'] = $defaultSort;
                }

                // Execute the query
                $queryBuilder = new QueryBuilder($model, $queryRequest);
                $data = $queryBuilder->get();
            }
        }

        // Get column configuration from PianolaList
        if ($sharedDataList->pianolaList && $sharedDataList->pianolaList->selection) {
            $selection = json_decode($sharedDataList->pianolaList->selection, true);

            // Filter out selection and action columns
            $columns = collect($selection)->filter(function ($column) {
                $text = strtolower($column['text'] ?? '');
                $value = strtolower($column['value'] ?? '');

                return ! in_array($text, ['selection', 'action']) && ! in_array($value, ['selection', 'action']);
            })->values()->toArray();
        }

        // Check if this is an API request
        if ($request->expectsJson() || $request->is('api/*')) {
            return response()->json([
                'shared_data_list' => $sharedDataList,
                'data' => $data,
                'columns' => $columns,
                'user' => $user,
            ]);
        }

        // Get app configuration
        $appConfig = $this->getAppConfig();

        // Set app locale
        app()->setLocale($appConfig['appLanguage']);

        // Check if calendar is available for this module
        $hasCalendar = $this->getHasCalendar($sharedDataList->pianolaList);

        // Check display mode
        $displayMode = $request->get('display', 'list');

        if ($displayMode === 'calendar' && $hasCalendar) {
            // Get calendar data mapping
            $calendarData = $this->getCalendarData($sharedDataList->pianolaQuery, $data);

            // For calendar view, only pass the record count instead of full data to reduce response size
            $recordCount = $data->count();

            return view('pianola-laravel::shared-data.calendar', compact('sharedDataList', 'recordCount', 'columns', 'user', 'appConfig', 'hasCalendar', 'calendarData'));
        }

        if ($displayMode === 'calendar_print' && $hasCalendar) {
            // Get calendar data for printing
            $calendarData = $this->getCalendarData($sharedDataList->pianolaQuery, $data);

            // Get print parameters
            $type = $request->get('type', 'daily'); // daily or weekly
            $date = $request->get('date', now()->format('Y-m-d'));

            // Filter events for the specified date/period
            $events = $this->filterEventsForPrint($calendarData, $type, $date);

            // Get locations for the filtered events
            $locations = $this->getLocationsForPrint($events);

            // Get app configuration for logo and title
            $logoPath = $appConfig['logoPath'] ?? null;
            $title = $this->getPrintTitle($date, $type);

            return view('pianola-laravel::printouts.calendar', compact('events', 'date', 'locations', 'title', 'logoPath', 'type'));
        }

        if ($displayMode === 'calendar_print_all' && $hasCalendar) {
            // Get calendar data for printing all periods
            $calendarData = $this->getCalendarData($sharedDataList->pianolaQuery, $data);

            // Get print parameters
            $type = $request->get('type', 'daily'); // daily or weekly

            // Get all available periods and their data
            $allPeriodsData = $this->getAllPeriodsForPrint($calendarData, $type);

            // Get app configuration for logo and title
            $logoPath = $appConfig['logoPath'] ?? null;
            $listName = $sharedDataList->name;

            // Set app locale for translations
            app()->setLocale($appConfig['appLanguage']);

            return view('pianola-laravel::printouts.calendar-all', compact('allPeriodsData', 'logoPath', 'type', 'listName', 'appConfig'));
        }

        // For table view, pre-filter data to only include columns that will be displayed
        // This reduces response size significantly by removing unused database fields
        $filteredData = $this->filterDataByColumns($data, $columns);
        $recordCount = $data->count();

        // Return default list view
        return view('pianola-laravel::shared-data.show', compact('sharedDataList', 'filteredData', 'recordCount', 'columns', 'user', 'appConfig', 'hasCalendar'));
    }

    /**
     * Generate an unlock token for a given email
     * This is a utility method that can be used to create unlock tokens
     */
    public function generateUnlockToken(Request $request)
    {
        $request->validate([
            'email' => 'required|email|exists:pianola_shared_data_users,email',
        ]);

        $token = Crypt::encryptString($request->email);

        return response()->json([
            'unlock_token' => $token,
        ]);
    }

    /**
     * Get app configuration from pianola/app.json
     */
    protected function getAppConfig()
    {
        if (! File::exists(config_path('pianola/app.json'))) {
            return [
                'colorPrimary' => '#007cba',
                'logoPath' => null,
                'appName' => 'Shared Data Portal',
            ];
        }

        $config = json_decode(File::get(config_path('pianola/app.json')), true);
        $basicConfig = $config['basicConfig'] ?? [];

        return [
            'colorPrimary' => $basicConfig['colorPrimary'] ?? '#007cba',
            'logoPath' => $basicConfig['logoPath'] ?? null,
            'appName' => $basicConfig['appName'] ?? 'Shared Data Portal',
            'appLanguage' => $basicConfig['appLanguage'] ?? 'en',
        ];
    }

    /**
     * Get module configuration for a PianolaList
     */
    protected function getModuleConfig($pianolaList)
    {
        if (! $pianolaList || ! $pianolaList->module) {
            return null;
        }

        if (! File::exists(config_path('pianola/app.json'))) {
            return null;
        }

        $config = json_decode(File::get(config_path('pianola/app.json')), true);
        $modules = $config['modules'] ?? [];

        // Find module by name (case-insensitive match)
        $listModuleName = strtolower($pianolaList->module);

        foreach ($modules as $module) {
            $basicConfig = $module['basicConfig'] ?? [];
            $moduleName = strtolower($basicConfig['name'] ?? '');

            if ($moduleName === $listModuleName) {
                return $basicConfig;
            }
        }

        return null;
    }

    /**
     * Get default sort order for a PianolaList from module configuration
     */
    protected function getDefaultSort($pianolaList)
    {
        $moduleConfig = $this->getModuleConfig($pianolaList);

        return $moduleConfig['defaultSort'] ?? null;
    }

    /**
     * Check if calendar is available for a PianolaList from module configuration
     */
    protected function getHasCalendar($pianolaList)
    {
        $moduleConfig = $this->getModuleConfig($pianolaList);

        return $moduleConfig['hasCalendar'] ?? false;
    }

    /**
     * Get calendar data by mapping fields from schema.json (similar to CalendarPrintoutController)
     */
    protected function getCalendarData($pianolaQuery, $data)
    {
        if (! $pianolaQuery || ! $pianolaQuery->base_table) {
            return [];
        }

        // Get calendar mapping from schema.json
        $calendarMapping = $this->getCalendarMapping($pianolaQuery->base_table);
        if (! $calendarMapping) {
            return [];
        }

        // Transform data to standardized calendar format (like CalendarPrintoutController)
        return $this->transformToCalendarEvents($data, $calendarMapping);
    }

    /**
     * Transform data to calendar events format (similar to CalendarPrintoutController)
     */
    protected function transformToCalendarEvents($data, $calendarMapping)
    {
        $map = [
            'id' => $calendarMapping['id'] ?? 'id',
            'date' => $calendarMapping['date'] ?? 'date',
            'timestamp_start' => $calendarMapping['timestamp_start'] ?? 'timestamp_start',
            'timestamp_end' => $calendarMapping['timestamp_end'] ?? 'timestamp_end',
            'location' => $calendarMapping['location'] ?? 'location_name',
            'name' => $calendarMapping['name'] ?? 'calendar_event_name',
            'more_information' => $calendarMapping['more_information'] ?? 'calendar_information',
            'locked' => $calendarMapping['locked'] ?? 'calendar_locked',
            'color' => $calendarMapping['color'] ?? 'calendar_color',
        ];

        return collect($data)->map(function ($record) use ($map) {
            // Skip records without required fields
            if (empty(Arr::get($record, $map['date']))) {
                return null;
            }
            if (empty(Arr::get($record, $map['timestamp_start']))) {
                return null;
            }

            // Set default end time if missing (15 minutes after start)
            $timestampEnd = Arr::get($record, $map['timestamp_end']);
            if (empty($timestampEnd)) {
                $startTime = Arr::get($record, $map['timestamp_start']);
                $timestampEnd = date('Y-m-d H:i:s', strtotime($startTime) + 900);
            }

            return [
                'id' => Arr::get($record, $map['id']),
                'date' => Arr::get($record, $map['date']),
                'calendar_id' => Arr::get($record, $map['id']),
                'calendar_timestamp_start' => Arr::get($record, $map['timestamp_start']),
                'calendar_timestamp_end' => $timestampEnd,
                'calendar_location' => Arr::get($record, $map['location']),
                'calendar_name' => Arr::get($record, $map['name']),
                'calendar_more_information' => Arr::get($record, $map['more_information']),
                'calendar_locked' => Arr::get($record, $map['locked']),
                'calendar_color' => $this->correctColor(Arr::get($record, $map['color'])),
            ];
        })->filter()->values()->toArray();
    }

    /**
     * Correct color format (from CalendarPrintoutController)
     */
    protected function correctColor($color)
    {
        if (empty($color)) {
            return '#ccc';
        }
        if (strpos($color, '#') === false) {
            return '#'.$color;
        }

        return $color;
    }

    /**
     * Get calendar mapping from schema.json for a specific table
     */
    protected function getCalendarMapping($tableName)
    {
        if (! File::exists(config_path('pianola/schema.json'))) {
            return null;
        }

        $schemas = json_decode(File::get(config_path('pianola/schema.json')), true);

        if (! is_array($schemas)) {
            return null;
        }

        // Find schema by table name
        foreach ($schemas as $schema) {
            if (isset($schema['name']) && $schema['name'] === $tableName) {
                return $schema['appCalendarMapping'] ?? null;
            }
        }

        return null;
    }

    /**
     * Filter data collection to only include columns that will be displayed
     * This reduces response size by removing unused database fields
     */
    protected function filterDataByColumns($data, $columns)
    {
        if (empty($columns)) {
            return $data;
        }

        // Extract column field names
        $columnFields = collect($columns)->pluck('value')->filter()->toArray();

        // Add 'id' field if not already included (needed for Vue key binding)
        if (! in_array('id', $columnFields)) {
            $columnFields[] = 'id';
        }

        // Filter each record to only include needed fields
        return $data->map(function ($record) use ($columnFields) {
            $filteredRecord = [];

            foreach ($columnFields as $field) {
                // Handle nested field access (e.g., 'user.name')
                if (strpos($field, '.') !== false) {
                    $value = $this->getNestedValue($record, $field);
                } else {
                    // Since QueryBuilder returns Eloquent models, use object property access
                    $value = $record->$field ?? null;
                }

                $filteredRecord[$field] = $value;
            }

            return $filteredRecord;
        });
    }

    /**
     * Get nested value from record (e.g., 'user.name' from record)
     * Handles Eloquent models and their relationships
     */
    protected function getNestedValue($record, $field)
    {
        $keys = explode('.', $field);
        $value = $record;

        foreach ($keys as $key) {
            if (is_object($value)) {
                // For Eloquent models, use property access or relationship loading
                if (isset($value->$key)) {
                    $value = $value->$key;
                } elseif (method_exists($value, $key)) {
                    // Try to load relationship if it's a method
                    try {
                        $value = $value->$key;
                    } catch (\Exception $e) {
                        return null;
                    }
                } else {
                    return null;
                }
            } elseif (is_array($value) && isset($value[$key])) {
                $value = $value[$key];
            } else {
                return null;
            }
        }

        return $value;
    }

    /**
     * Filter events for print based on type and date
     */
    protected function filterEventsForPrint($calendarData, $type, $date)
    {
        if ($type === 'daily') {
            return collect($calendarData)->filter(function ($event) use ($date) {
                return $event['date'] === $date;
            })->values()->toArray();
        } else {
            // Weekly view - get events for the week containing the date
            $mondayBefore = date('Y-m-d', strtotime($date.' - '.date('w', strtotime($date)) - 1 .' days'));
            $weekEnd = date('Y-m-d', strtotime($mondayBefore.' + 6 days'));

            return collect($calendarData)->filter(function ($event) use ($mondayBefore, $weekEnd) {
                return $event['date'] >= $mondayBefore && $event['date'] <= $weekEnd;
            })->values()->toArray();
        }
    }

    /**
     * Get unique locations from filtered events
     */
    protected function getLocationsForPrint($events)
    {
        return collect($events)
            ->pluck('calendar_location')
            ->filter()
            ->unique()
            ->values()
            ->toArray();
    }

    /**
     * Generate print title based on date and type
     */
    protected function getPrintTitle($date, $type)
    {
        $date = date('d.m.Y', strtotime($date));
        $weekday = match (date('l', strtotime($date))) {
            'Monday' => 'Montag',
            'Tuesday' => 'Dienstag',
            'Wednesday' => 'Mittwoch',
            'Thursday' => 'Donnerstag',
            'Friday' => 'Freitag',
            'Saturday' => 'Samstag',
            'Sunday' => 'Sonntag',
        };

        if ($type === 'daily') {
            return 'Tagesübersicht '.$weekday.', '.$date;
        }

        $mondayBefore = date('Y-m-d', strtotime($date.' - '.date('w', strtotime($date)) - 1 .' days'));

        return 'Wochenübersicht: '.date('d.m.Y', strtotime($mondayBefore)).' - '.date('d.m.Y', strtotime($mondayBefore.' + 6 days'));
    }

    /**
     * Get all periods data for printing (all days or all weeks)
     */
    protected function getAllPeriodsForPrint($calendarData, $type)
    {
        $allPeriods = [];

        if ($type === 'daily') {
            // Get all unique dates
            $dates = collect($calendarData)->pluck('date')->unique()->sort()->values();

            foreach ($dates as $date) {
                $events = collect($calendarData)->filter(function ($event) use ($date) {
                    return $event['date'] === $date;
                })->values()->toArray();

                if (! empty($events)) {
                    $locations = $this->getLocationsForPrint($events);
                    $title = $this->getPrintTitle($date, 'daily');

                    $allPeriods[] = [
                        'date' => $date,
                        'events' => $events,
                        'locations' => $locations,
                        'title' => $title,
                        'type' => 'daily',
                    ];
                }
            }
        } else {
            // Get all unique weeks
            $dates = collect($calendarData)->pluck('date')->unique()->sort()->values();
            $weeks = [];

            foreach ($dates as $date) {
                $mondayBefore = date('Y-m-d', strtotime($date.' - '.date('w', strtotime($date)) - 1 .' days'));
                if (! in_array($mondayBefore, $weeks)) {
                    $weeks[] = $mondayBefore;
                }
            }

            foreach ($weeks as $weekStart) {
                $weekEnd = date('Y-m-d', strtotime($weekStart.' + 6 days'));

                $events = collect($calendarData)->filter(function ($event) use ($weekStart, $weekEnd) {
                    return $event['date'] >= $weekStart && $event['date'] <= $weekEnd;
                })->values()->toArray();

                if (! empty($events)) {
                    $locations = $this->getLocationsForPrint($events);
                    $title = $this->getPrintTitle($weekStart, 'weekly');

                    $allPeriods[] = [
                        'date' => $weekStart,
                        'events' => $events,
                        'locations' => $locations,
                        'title' => $title,
                        'type' => 'weekly',
                    ];
                }
            }
        }

        return $allPeriods;
    }
}
