<?php

namespace BalletMecanique\PianolaLaravel\Services;

use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Str;

class SchemaUpdateService
{
    protected $schema;

    public function __construct()
    {
        $this->schema = json_decode(File::get(config_path('/pianola/schema.json')), true);
    }

    public function updateSchema()
    {
        $tables = $this->getDatabaseSchema();

        $schemaCollection = collect($this->schema);

        $schemaCollection = $this->addMissingTables($tables, $schemaCollection);
        $schemaCollection = $this->sortSchema($schemaCollection);
        $schemaCollection = $this->addMissingColumns($tables, $schemaCollection);
        $schemaCollection = $this->addMissingModelNames($schemaCollection);
        $schemaCollection = $this->addMissingRelationships($schemaCollection);
        $schemaCollection = $this->sortKeysWithinSchemaTable($schemaCollection);

        return $schemaCollection;

        return collect($schemaCollection)->filter()->values()->all();
    }

    protected function getDatabaseSchema()
    {
        return collect(DB::select('SHOW TABLES'))
            ->map(function ($table) {
                $tableName = reset($table);

                return [
                    'tableName' => $tableName,
                    'columns' => collect(DB::select('SHOW COLUMNS FROM '.$tableName))->map(function ($column) {
                        return [
                            'name' => $column->Field,
                            'column_type' => $column->Type,
                            'app_label' => '',
                        ];
                    }),
                ];
            });
    }

    public function addMissingTables($tables, $schemaCollection)
    {
        foreach ($tables as $table) {
            $avoidTables = ['password_resets', 'failed_jobs', 'personal_access_tokens'];
            if (
                in_array($table['tableName'], $avoidTables)
                or
                strpos($table['tableName'], 'pianola') !== false
            ) {
                continue;
            }
            // add table if not yet existent
            if (
                $schemaCollection->where('name', $table['tableName'])->count() < 1
            ) {
                $schemaCollection[] = [
                    'name' => $table['tableName'],
                    'columns' => $table['columns'],
                ];
            }
        }

        return collect($schemaCollection);
    }

    protected function sortSchema($schemaCollection)
    {
        $schemaCollection = $schemaCollection->sortBy('name')->values()->all();

        return collect($schemaCollection);
    }

    protected function addMissingColumns($tables, $schemaCollection)
    {
        return $schemaCollection->map(function ($schemaCollectionTable) use ($tables) {
            $table = collect($tables)->where('tableName', $schemaCollectionTable['name'])->values()->first();
            $tableColumns = $table['columns'];
            foreach ($tableColumns as $tableColumn) {
                if (
                    collect($schemaCollectionTable['columns'])->where('name', $tableColumn['name'])->count() < 1
                ) {
                    $schemaCollectionTable['columns'][] = $tableColumn;
                }
            }
            $sortedColumns = [];
            foreach ($tableColumns as $tableColumn) {
                $columnInSchema = collect($schemaCollectionTable['columns'])->firstWhere('name', $tableColumn['name']);
                $sortedColumns[] = $columnInSchema;
            }
            $schemaCollectionTable['columns'] = $sortedColumns;

            return $schemaCollectionTable;
        });
    }

    protected function addMissingRelationships($schemaCollection)
    {
        return $schemaCollection->map(function ($schemaTable) use ($schemaCollection) {
            $belongsToTables = $this->getBelongsToRelationships($schemaTable);
            $newBelongsToArray = [];
            foreach ($belongsToTables as $belongsToTable) {
                if (collect($schemaTable['belongsTo'] ?? [])->where('name', $belongsToTable)->count() < 1) {
                    $newBelongsToArray[] = ['name' => $belongsToTable];
                } else {
                    $newBelongsToArray[] = collect($schemaTable['belongsTo'])->where('name', $belongsToTable)->values()->first();
                }
            }
            $schemaTable['belongsTo'] = $newBelongsToArray;

            $hasManyTables = $this->getHasManyRelationships($schemaTable, $schemaCollection);
            $newHasManyArray = [];
            foreach ($hasManyTables as $hasManyTable) {
                if (collect($schemaTable['hasMany'] ?? [])->where('name', $hasManyTable)->count() < 1) {
                    $newHasManyArray[] = ['name' => $hasManyTable];
                } else {
                    $newHasManyArray[] = collect($schemaTable['hasMany'])->where('name', $hasManyTable)->values()->first();
                }
            }
            $schemaTable['hasMany'] = $newHasManyArray;

            $schemaTable['belongsTo'] = collect($schemaTable['belongsTo'])->sortBy('name')->values()->all();
            $schemaTable['hasMany'] = collect($schemaTable['hasMany'])->sortBy('name')->values()->all();

            return $schemaTable;
        });
    }

    protected function addMissingModelNames($schemaCollection)
    {
        return $schemaCollection->map(function ($schemaTable) {
            $modelName = Str::singular(Str::studly($schemaTable['name']));
            if (! isset($schemaTable['modelName'])) {
                $schemaTable['modelName'] = $modelName;
            }

            return $schemaTable;
        });
    }

    protected function getBelongsToRelationships($schemaTable)
    {
        return collect($schemaTable['columns'])->map(function ($column) {
            if (Str::endsWith($column['name'], '_id')) {
                $modelName = Str::studly(Str::replace('_id', '', $column['name']));
                if ($this->getTableNameFromModelName($modelName)) {
                    return $this->getTableNameFromModelName($modelName);
                }
            }
        })->filter()->values();
    }

    protected function getHasManyRelationships($table, $schema)
    {
        $foreignKeyFieldName = Str::singular($table['name']).'_id';

        return collect($schema)->map(function ($table) use ($foreignKeyFieldName) {
            if (collect($table['columns'])->where('name', $foreignKeyFieldName)->count() > 0) {
                if ($this->tableNameExistsInSchema($table['name'])) {
                    return $table['name'];
                }
            }
        })->filter()->values();
    }

    public function createModels()
    {
        $models = collect($this->schema)->pluck('modelName');
        $models->each(function ($model) {
            $modelPath = 'Models/'.$model.'.php';
            if (! file_exists(app_path($modelPath))) {
                $this->createFile(app_path($modelPath), $this->getModelString($model));
            }
        });

        return 'ok';
    }

    protected function tableNameExistsInSchema($tableName)
    {
        return collect($this->schema)->where('name', $tableName)->count() > 0;
    }

    protected function createFile($filepath, $content)
    {
        if (! file_exists($filepath)) {
            File::put($filepath, $content);
        }
    }

    protected function getModelString($model)
    {
        $tableName = $this->getTableNameFromModelName($model);
        $pluralisedModelName = Str::snake(Str::plural($model));
        $showModelString = $tableName !== $pluralisedModelName;
        $tableString = $showModelString ? ("\n".'    protected $table = "'.$tableName.'";') : '';

        return <<<EOT
        <?php

        namespace App\Models;

        use Illuminate\Database\Eloquent\Factories\HasFactory;
        use Illuminate\Database\Eloquent\Model;

        class $model extends Model
        {
            use HasFactory;

            protected \$guarded = [];$tableString

        }

        EOT;
    }

    protected function getTableNameFromModelName($modelName)
    {
        return collect($this->schema)->where('modelName', $modelName)->first()['name'] ?? '';
    }

    protected function sortKeysWithinSchemaTable($schemaCollection)
    {
        $sortOrder = [
            'name',
            'modelName',
            'app_name_singular',
            'app_record_filter_columns',
            'columns',
            'custom_attributes',
            'belongsTo',
            'hasMany',
        ];

        return $schemaCollection->map(function ($schemaTable) use ($sortOrder) {
            $sortedSchemaTable = [];
            foreach ($sortOrder as $key) {
                if (isset($schemaTable[$key])) {
                    $sortedSchemaTable[$key] = $schemaTable[$key];
                }
            }
            $otherKeys = collect($schemaTable)->keys()->diff(collect($sortOrder));
            foreach ($otherKeys as $key) {
                $sortedSchemaTable[$key] = $schemaTable[$key];
            }

            return $sortedSchemaTable;
        });
    }
}
