<?php

namespace BalletMecanique\PianolaLaravel\Http\Controllers;

use BalletMecanique\PianolaLaravel\Services\PianolaUserState;
use BalletMecanique\PianolaLaravel\Services\QueryBuilder;
use BalletMecanique\PianolaLaravel\Services\RelationshipBuilder;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Str;

class RecordsController
{
    public function count($endpoint, Request $request)
    {
        $modelName = 'App\\Models\\'.Str::singular(Str::studly($endpoint));
        $model = new $modelName();

        return (new QueryBuilder($model, $request))->count();
    }

    public function index($endpoint, Request $request)
    {
        $this->checkRights($endpoint);
        $modelName = 'App\\Models\\'.Str::singular(Str::studly($endpoint));
        $model = new $modelName();
        $records = new QueryBuilder($model, $request);
        $records = $records->get();
        $loadAllRelationships = $request->input('load') === 'true' || $request->input('excel') === 'true';
        $relationships = $this->getRelationshipLoadArray($endpoint, $loadAllRelationships);
        $this->resolveRelationships($relationships, $endpoint);
        $this->resolvePolymorphicFileTagRelationships($endpoint);
        $records->load($relationships);

        if (class_exists('App\\Models\\FileTag')) {
            $records->load('file_tags.file');
        }

        $records = $this->appendCustomAttributes($endpoint, $records, $loadAllRelationships);

        $records->each(function ($record) use ($endpoint) {
            $this->appendManualAppends($record, $endpoint);
        });

        if (! $loadAllRelationships) {
            $records = $this->removeHasManyRelatedRecords($records, $relationships);
        }

        if ($request->input('calendar') === 'true') {
            $records = $this->returnOnlyCalendarData($records, $endpoint);
        }

        if ($request->input('excel') === 'true') {
            $records = $this->returnOnlyRequestedHeaders($records, $request->input('headers'));
        }

        if ($request->input('load') === 'true') {
            $records = $this->limitResponseSize($records, $relationships, 5);
        }

        return $records;
    }

    public function show($endpoint, $id)
    {
        $this->checkRights($endpoint);
        if ($id === 'null') {
            return [];
        }
        $modelName = 'App\\Models\\'.Str::singular(Str::studly($endpoint));
        $model = $modelName::find($id);
        $relationships = $this->getRelationshipLoadArray($endpoint, true);
        $this->resolveRelationships($relationships, $endpoint);
        $this->resolvePolymorphicFileTagRelationships($endpoint);
        $model->load($relationships);
        if (class_exists('App\\Models\\FileTag')) {
            $model->load('file_tags.file');
        }
        $model = $this->appendCustomAttributes($endpoint, $model, true);
        $model = $this->appendManualAppends($model, $endpoint);

        return $model;
    }

    public function store($endpoint, Request $request)
    {
        $modelName = 'App\\Models\\'.Str::singular(Str::studly($endpoint));
        $model = new $modelName();
        $maxId = $model::max('id');
        $model->fill($request->input('data'));
        $model->id = $maxId + 1;
        $model->save();
        if ($request->input('load')) {
            return $this->show($endpoint, $maxId + 1);
        }

        return $model;
    }

    public function update($endpoint, $id, Request $request)
    {
        $modelName = 'App\\Models\\'.Str::singular(Str::studly($endpoint));
        $record = $modelName::find($id);
        if ($record->exists()) {
            $data = $request->input('data');
            //strip out any keys with a prefix of @, : or #
            $nonStateData = collect($data)->filter(function ($value, $key) {
                return ! in_array(substr($key, 0, 1), ['@', ':', '#']);
            })->all();
            $stateData = collect($data)->filter(function ($value, $key) {
                return in_array(substr($key, 0, 1), ['@', ':', '#']);
            })->all();
            if(! empty($stateData)) {
                PianolaUserState::storeStateData($stateData, $endpoint, $id);
            }
            $record->update($nonStateData);
            if ($request->input('load')) {
                return $this->show($endpoint, $id);
            }

            return 'ok';
        }

        return response('record does not exist', 404);
    }

    public function destroy($endpoint, $id)
    {
        $modelName = 'App\\Models\\'.Str::singular(Str::studly($endpoint));
        $modelName::destroy($id);

        return $id;
    }

    protected function resolveRelationships($relationshipLoadArray, $endpoint)
    {
        foreach ($relationshipLoadArray as $relationship) {
            $pathArray = explode('.', $endpoint.'.'.$relationship);
            foreach ($pathArray as $key => $origin) {
                if ($key !== array_key_last($pathArray)) {
                    $originModelName = 'App\\Models\\'.Str::singular(Str::studly($origin));
                    if (! class_exists($originModelName)) {
                        continue;
                    }
                    $relationship = $pathArray[$key + 1];
                    $originModelName::resolveRelationUsing($relationship, function ($originModel) use ($relationship, $origin) {
                        $modelName = $this->checkIfModelName(Str::studly($relationship)) ? Str::studly($relationship) : Str::singular(Str::studly($relationship));
                        $relatedModel = 'App\\Models\\'.$modelName;
                        if (Str::singular($relationship) == $relationship || $this->checkIfModelName($relationship)) {
                            return $originModel->belongsTo($relatedModel, $relationship.'_id');
                        }

                        $resolvedRelationship = $originModel->hasMany($relatedModel);
                        if ($sortRules = $this->getSortRules($origin, $relationship)) {
                            foreach ($sortRules as $sortRule) {
                                $resolvedRelationship = $resolvedRelationship->orderBy($sortRule['field'], $sortRule['sortOrder'] ?? 'asc');
                            }
                        }

                        return $resolvedRelationship;
                    });
                }
            }
        }
    }

    protected function resolvePolymorphicFileTagRelationships($endpoint)
    {
        //if exists FileTag model, resolve polymorphic relationship
        if (class_exists('App\\Models\\FileTag')) {
            $modelName = 'App\\Models\\'.Str::singular(Str::studly($endpoint));
            $modelName::resolveRelationUsing('file_tags', function ($model) {
                return $model->morphMany('App\\Models\\FileTag', 'fileable')
                    ->orderByRaw('sort_order IS NULL, sort_order')
                    ->orderBy('created_at', 'desc');
            });
        }
    }

    protected function getRelationshipLoadArray($endpoint, $loadAllRelationships)
    {
        $schemaRelationships = $this->getSchemaRelationshipsLoadArray($endpoint, $loadAllRelationships);
        $customAttributeRelationships = $this->getCustomAttributesLoadArray($endpoint);
        $loadArray = collect($schemaRelationships)
            ->merge($customAttributeRelationships)
            ->unique()
            ->values()
            ->all();

        return $loadArray;
    }

    protected function getSchemaRelationshipsLoadArray($endpoint, $loadAllRelationships)
    {
        $relationships = new RelationshipBuilder($endpoint, null);
        $hasManyLoadArray = $loadAllRelationships ? $relationships->getHasManyLoadArray() : [];
        $belongsToLoadArray = $relationships->getBelongsToLoadArray();
        $extraLoadArray = $relationships->getExtraLoadArray();
        $loadArray = collect($hasManyLoadArray)
            ->merge($belongsToLoadArray)
            ->merge($extraLoadArray)
            ->unique()
            ->values()
            ->all();

        return $loadArray;
    }

    protected function getCustomAttributesLoadArray($endpoint)
    {
        $relationships = new RelationshipBuilder($endpoint, null);
        $appendLoadArray = $relationships->getAppendLoadArray();
        $loadArray = collect($appendLoadArray)
            ->unique()
            ->values()
            ->all();

        return $loadArray;
    }

    protected function getSortRules($origin, $relationship)
    {
        $schema = collect(json_decode(File::get(config_path('/pianola/schema.json')), true));
        $tableSchema = collect($schema->where('name', $origin)->first());

        return
            collect($tableSchema['hasMany'] ?? [])->where('name', $relationship)->first()['sortOrder']
            ??
            collect($tableSchema['belongsTo'] ?? [])->where('name', $relationship)->first()['sortOrder']
            ??
            null;
    }

    protected function checkIfModelName($relationship)
    {
        $schema = collect(json_decode(File::get(config_path('/pianola/schema.json')), true));

        return $schema->where('modelName', Str::title($relationship))->count() > 0;
    }

    protected function appendCustomAttributes($endpoint, $records, $loadAllRelationships)
    {
        $schema = json_decode(File::get(config_path('/pianola/schema.json')), true);
        $customAttributes = collect(collect($schema)->where('name', $endpoint)->pluck('custom_attributes')->first())
            ->map(function ($customAttribute) {
                return $customAttribute['name'] ?? null;
            })->filter()->unique()->toArray();

        return $this->appendAttributesToRecords($records, $customAttributes);
    }

    protected function appendAttributesToRecords($records, $attributes)
    {
        foreach ($attributes as $attribute) {
            if (strpos($attribute, '.') !== false) {
                // If the attribute contains dot notation, we need to handle relationships
                $parts = explode('.', $attribute);
                $relationship = array_shift($parts);
                $remainingAttribute = implode('.', $parts);

                // Process each record individually
                if ($records instanceof \Illuminate\Database\Eloquent\Collection) {
                    $records->each(function ($record) use ($relationship, $remainingAttribute) {
                        $this->appendAttributesToRecord($record, $relationship, $remainingAttribute);
                    });
                } elseif ($records instanceof \Illuminate\Database\Eloquent\Model) {
                    $this->appendAttributesToRecord($records, $relationship, $remainingAttribute);
                }
            } else {
                // Append attribute to the root record
                $records->append($attribute);
            }
        }

        return $records;
    }

    protected function appendAttributesToRecord($record, $relationship, $remainingAttribute)
    {
        $relatedRecords = $record->$relationship;

        if ($relatedRecords instanceof \Illuminate\Database\Eloquent\Collection) {
            // Append attributes to each related record in the collection
            $relatedRecords->each(function ($relatedRecord) use ($remainingAttribute) {
                $this->appendAttributesToRecords($relatedRecord, [$remainingAttribute]);
            });
        } elseif ($relatedRecords instanceof \Illuminate\Database\Eloquent\Model) {
            // Append attributes to a single related record
            $this->appendAttributesToRecords($relatedRecords, [$remainingAttribute]);
        }
    }

    protected function appendManualAppends($model, $endpoint)
    {
        $schema = collect(json_decode(File::get(config_path('/pianola/schema.json')), true));
        $string = collect($schema)->where('name', $endpoint)->pluck('append')->first();
        $manualAppendArray = $string ? explode(',', $string) : null;
        if ($manualAppendArray) {
            $appendCollection = collect($manualAppendArray);
            $directAppends = $appendCollection->filter(function ($attribute) {
                return strpos($attribute, '.') == 0;
            });
            $indirectAppends = $appendCollection->filter(function ($attribute) {
                return strpos($attribute, '.') != false;
            });
            $model->append($directAppends->all());
            $indirectAppends->each(function ($indirectAppend) use ($model) {
                $relatedModel = strtok($indirectAppend, '.');
                $appendedAttribute = str_replace($relatedModel.'.', '', $indirectAppend);
                $model->$relatedModel?->each->append($appendedAttribute);
            });
        }

        return $model;
    }

    protected function removeHasManyRelatedRecords($records, $relationships)
    {
        $relationshipsToBeHidden = collect($relationships)->map(function ($relationship) {
            $relationship = explode('.', $relationship)[0];
            if (Str::singular($relationship) == $relationship) {
                return null;
            }

            return $relationship;
        })->filter()->unique()->toArray();

        return $records->makeHidden($relationshipsToBeHidden);
    }

    protected function limitResponseSize($array, $relationships, $maxSizeInMb)
    {
        $sizeInBytes = $maxSizeInMb * 1048576;
        $relationships = collect($relationships)->map(function ($relationship) {
            return explode('.', $relationship)[0];
        })->unique()->toArray();
        $size = strlen(json_encode($array));
        if ($size > $sizeInBytes) {
            $sizeMap = [];
            foreach ($relationships as $relationship) {
                $values = collect($array)->pluck($relationship);
                $sizeOfValues = strlen(json_encode($values));
                $sizeMap[] = ['key' => $relationship, 'size' => $sizeOfValues];
            }
            usort($sizeMap, function ($a, $b) {
                return $a['size'] <=> $b['size'];
            });
            $key = array_pop($sizeMap)['key'];
            $array->transform(function ($record) use ($key) {
                unset($record[$key]);

                return $record;
            });
        }

        return $array;
    }

    protected function returnOnlyRequestedHeaders($records, $headers)
    {
        $fields = explode('|', $headers);

        return $records->map(function ($record) use ($fields) {
            $returnedRecord = [];
            foreach ($fields as $field) {
                if (str_contains($field, '.0.')) {
                    $tableName = (strtok($field, '.0.'));
                    $fieldName = str_replace($tableName.'.0.', '', $field);
                    $returnedRecord[$tableName][0][$fieldName] = $record[$tableName][0][$fieldName] ?? null;
                } elseif (str_contains($field, '.')) {
                    $tableName = (strtok($field, '.'));
                    $fieldName = str_replace($tableName.'.', '', $field);
                    $returnedRecord[$tableName][$fieldName] = $record[$tableName][$fieldName] ?? null;
                } else {
                    $returnedRecord[$field] = $record[$field];
                }
            }

            return $returnedRecord;
        });
    }

    protected function returnOnlyCalendarData($records, $endpoint)
    {
        $schema = collect(json_decode(File::get(config_path('/pianola/schema.json')), true));
        $calendarMapping = collect($schema->where('name', $endpoint)->first()['appCalendarMapping'] ?? []);
        $map = [
            'id' => $calendarMapping['id'] ?? 'id',
            'date' => $calendarMapping['date'] ?? 'date',
            'timestamp_start' => $calendarMapping['timestamp_start'] ?? 'timestamp_start',
            'timestamp_end' => $calendarMapping['timestamp_end'] ?? 'timestamp_end',
            'location' => $calendarMapping['location'] ?? 'location_name',
            'name' => $calendarMapping['name'] ?? 'calendar_event_name',
            'more_information' => $calendarMapping['more_information'] ?? 'calendar_information',
            'locked' => $calendarMapping['locked'] ?? 'calendar_locked',
            'color' => $calendarMapping['color'] ?? 'calendar_color',
        ];

        return $records->map(function ($record) use ($map) {
            if (empty($record[$map['date']])) {
                return null;
            }
            if (empty($record[$map['timestamp_start']])) {
                return null;
            }
            if (empty($record[$map['timestamp_end']])) {
                $record[$map['timestamp_end']] = date('Y-m-d H:i:s', strtotime($record[$map['timestamp_start']]) + 900);
            }

            return [
                'date' => $record[$map['date']],
                'calendar_id' => $record[$map['id']],
                'calendar_timestamp_start' => $record[$map['timestamp_start']],
                'calendar_timestamp_end' => $record[$map['timestamp_end']],
                'calendar_location' => $record[$map['location']],
                'calendar_name' => $record[$map['name']],
                'calendar_more_information' => $record[$map['more_information']],
                'calendar_locked' => $record[$map['locked']],
                'calendar_color' => $this->correctColor($record[$map['color']]),
            ];
        })->filter()->values();
    }

    protected function correctColor($color)
    {
        if (empty($color)) {
            return '#ccc';
        }
        if (strpos($color, '#') === false) {
            return '#'.$color;
        }

        return $color;
    }

    protected function checkRights($endpoint)
    {
        $user = auth()->user();
        $userModules = json_decode($user->modules, true);

        //global rights set
        if ($userModules === []) {
            return null;
        }

        $appConfig = file_get_contents(config_path('pianola/app.json'));
        $configModules = collect(json_decode($appConfig, true)['modules']);
        $thisModuleName = $configModules->pluck('basicConfig')->where('apiEndpoint', $endpoint)->first()['name'] ?? null;

        //module name not in config: pass
        if (! $thisModuleName) {
            return null;
        }

        $result = $userModules[$thisModuleName] ?? (in_array($thisModuleName, $userModules) ? 'write' : 'none');
        if ($result === 'none') {
            return response()->json([
                'message' => 'You do not have access to this endpoint.',
            ], 403);
        }

        return null;
    }
}
