<?php

namespace BalletMecanique\PianolaLaravel\Services;

use Illuminate\Support\Arr;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use JsonException;

/**
 * Class PianolaUserState
 * This class is used to store and retrieve user state data for the current request.
 * The state data is stored in the database and is specific to the authenticated user.
 * The class was created as a work-around to storing user state data in the session,
 * which was unreliable in Laravel Vapor-based deployments.
 */

class PianolaUserState
{
    /**
     * Cached user state for the current request.
     */
    private static ?array $cachedState = null;

    /**
     * Find and cache the user's state for the current request.
     * @throws JsonException
     */
    public static function find(): void
    {
        if (is_null(self::$cachedState)) {
            $user_id = auth()->id();
            $record = DB::table('pianola_user_states')->where('user_id', $user_id)->first();
            $stateJson = $record->state ?? '{}';
            self::$cachedState = json_decode($stateJson, true, 512, JSON_THROW_ON_ERROR);
        }
    }

    /**
     * @throws JsonException
     */
    public static function store(): void
    {
        $user_id = auth()->id();
        $stateJson = json_encode(self::$cachedState, JSON_THROW_ON_ERROR);

        $existingRecord = DB::table('pianola_user_states')->where('user_id', $user_id)->exists();

        if ($existingRecord) {
            DB::table('pianola_user_states')->where('user_id', $user_id)->update([
                'state' => $stateJson,
                'updated_at' => now(),
            ]);
        } else {
            DB::table('pianola_user_states')->insert([
                'user_id' => $user_id,
                'state' => $stateJson,
                'created_at' => now(),
                'updated_at' => now(),
            ]);
        }
    }

    /**
     * Get a specific key from the user's state.
     * @throws JsonException
     */
    public static function get($key): mixed
    {
        self::find();

        return Arr::get(self::$cachedState, $key, null);
    }

    public static function getRecordStateValue($endpoint, $recordId, $key): mixed
    {
        return self::get($endpoint . '.' . $recordId . '.' . $key);
    }

    /**
     * Get a specific key from the user's state.
     * @throws JsonException
     */
    public static function getStateDataForRecord($endpoint, $recordId): mixed
    {
        self::find();
        //GLOBAL STATE
        $globalState = Arr::get(self::$cachedState, 'globalState');
        //attach @ to each $globalState key
        $globalState = collect($globalState)->mapWithKeys(function ($value, $key) {
            return ['@' . $key => $value];
        })->toArray();
        //ENDPOINT STATE
        $endpointState = Arr::get(self::$cachedState, $endpoint);
        //attach : to each $endpointState key
        $endpointState = collect($endpointState)->mapWithKeys(function ($value, $key) {
            return [':' . $key => $value];
        })->toArray();
        //remove all keys that have an array as a value
        $endpointState = collect($endpointState)->filter(function ($value) {
            return ! is_array($value);
        })->toArray();
        //RECORD STATE
        $recordState = Arr::get(self::$cachedState, $endpoint . '.' . $recordId);
        //attach # to each $recordState key
        $recordState = collect($recordState)->mapWithKeys(function ($value, $key) {
            return ['#' . $key => $value];
        })->toArray();
        $allStateData = array_merge($globalState ?? [], $endpointState ?? [], $recordState ?? []);

        return $allStateData;
    }

    public static function storeStateData($stateData, $endpoint, $recordId): void
    {
        collect($stateData)
             ->each(function ($value, $key) use ($endpoint, $recordId) {
                 $keyWithoutPrefix = substr($key, 1);
                 if(Str::startsWith($key, '@')) {
                     $path = 'globalState' . '.'. $keyWithoutPrefix;
                 } elseif (Str::startsWith($key, ':')) {
                     $path = $endpoint . '.'. $keyWithoutPrefix;
                 } elseif (Str::startsWith($key, '#')) {
                     $path = $endpoint . '.'. $recordId . '.' . $keyWithoutPrefix;
                 } else {
                     return;
                 }
                 PianolaUserState::set($path, $value);
             });
    }

    /**
     * @throws JsonException
     */
    public static function set($key, $value): void
    {
        self::find();
        if($value === null) {
            self::forget($key);

            return;
        }
        Arr::set(self::$cachedState, $key, $value);
        self::store();
    }

    public static function setRecordStateValue($endpoint, $recordId, $key, $value): void
    {
        self::set($endpoint . '.' . $recordId . '.' . $key, $value);
    }

    /**
     * @throws JsonException
     */
    public static function forget($key): void
    {
        self::find();
        Arr::forget(self::$cachedState, $key);
        $newKey = self::removeLastNodeFromKey($key);
        if ($newKey && empty(Arr::get(self::$cachedState, $newKey))) {
            self::forget($newKey);
        } else {
            self::store();
        }
    }

    public static function forgetRecordStateValue($endpoint, $recordId, $key): void
    {
        self::forget($endpoint . '.' . $recordId . '.' . $key);
    }

    public static function removeLastNodeFromKey($key)
    {
        $keyArray = explode('.', $key);
        array_pop($keyArray);

        return implode('.', $keyArray);
    }
}
