<?php

namespace BalletMecanique\PianolaLaravel\Http\Controllers;

use BalletMecanique\PianolaLaravel\Services\QueryBuilder;
use BalletMecanique\PianolaLaravel\Services\RelationshipBuilder;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Str;

class RecordsController
{
    public function count($endpoint, Request $request)
    {
        $modelName = 'App\\Models\\' . Str::singular(Str::studly($endpoint));
        $model = new $modelName();
        $records = new QueryBuilder($model, $request);

        return $records->count();
    }

    public function index($endpoint, Request $request)
    {
        $modelName = 'App\\Models\\' . Str::singular(Str::studly($endpoint));
        $model = new $modelName();
        $records = new QueryBuilder($model, $request);
        $records = $records->get();
        $loadAllRelationships = $request->input('load') === 'true' || $request->input('excel') === 'true';
        $relationships = $this->getRelationshipLoadArray($endpoint, $loadAllRelationships);
        $this->resolveRelationships($relationships, $endpoint);
        $this->resolvePolymorphicFileTagRelationships($endpoint);
        $records->load($relationships);
        if (class_exists('App\\Models\\FileTag')) {
            $records->load('file_tags.file');
        }
        $records = $this->appendCustomAttributes($endpoint, $records, $loadAllRelationships);
        if (! $loadAllRelationships) {
            $records = $this->removeHasManyRelatedRecords($records, $relationships);
        }
        if ($request->input('excel') === 'true') {
            $records = $this->returnOnlyRequestedHeaders($records, $request->input('headers'));
        } elseif ($request->input('load') === 'true') {
            $records = $this->limitResponseSize($records, $relationships, 5);
        }

        return $records;
    }

    public function show($endpoint, $id)
    {
        $modelName = 'App\\Models\\' . Str::singular(Str::studly($endpoint));
        $model = $modelName::find($id);
        $relationships = $this->getRelationshipLoadArray($endpoint, true);
        $this->resolveRelationships($relationships, $endpoint);
        $this->resolvePolymorphicFileTagRelationships($endpoint);
        $model->load($relationships);
        if (class_exists('App\\Models\\FileTag')) {
            $model->load('file_tags.file');
        }
        $model = $this->appendCustomAttributes($endpoint, $model, true);

        return $model;
    }

    public function store($endpoint, Request $request)
    {
        $modelName = 'App\\Models\\'.Str::singular(Str::studly($endpoint));
        $model = new $modelName();
        $maxId = $model::max('id');
        $model->fill($request->input('data'));
        $model->id = $maxId + 1;
        $model->save();
        if ($request->input('load')) {
            return $this->show($endpoint, $maxId + 1);
        } else {
            return 'ok';
        }
    }

    public function update($endpoint, $id, Request $request)
    {
        $modelName = 'App\\Models\\'.Str::singular(Str::studly($endpoint));
        $record = $modelName::find($id);
        if ($record->exists()) {
            $record->update($request->input('data'));
            if ($request->input('load')) {
                return $this->show($endpoint, $id);
            } else {
                return 'ok';
            }
        } else {
            return response('record does not exist', 404);
        }
    }

    public function destroy($endpoint, $id)
    {
        $modelName = 'App\\Models\\'.Str::singular(Str::studly($endpoint));
        $modelName::destroy($id);

        return $id;
    }

    protected function resolveRelationships($relationshipLoadArray, $endpoint)
    {
        foreach ($relationshipLoadArray as $relationship) {
            $pathArray = explode('.', $endpoint . '.'. $relationship);
            foreach ($pathArray as $key => $origin) {
                if ($key !== array_key_last($pathArray)) {
                    $originModelName = 'App\\Models\\' . Str::singular(Str::studly($origin));
                    if (! class_exists($originModelName)) {
                        continue;
                    }
                    $relationship = $pathArray[$key + 1];
                    $originModelName::resolveRelationUsing($relationship, function ($originModel) use ($relationship, $origin) {
                        $modelName = $this->checkIfModelName(Str::studly($relationship)) ? Str::studly($relationship) : Str::singular(Str::studly($relationship));
                        $relatedModel = 'App\\Models\\' . $modelName;
                        if (Str::singular($relationship) == $relationship || $this->checkIfModelName($relationship)) {
                            return $originModel->belongsTo($relatedModel, $relationship . '_id');
                        } else {
                            $resolvedRelationship = $originModel->hasMany($relatedModel);
                            if ($sortRules = $this->getSortRules($origin, $relationship)) {
                                foreach ($sortRules as $sortRule) {
                                    $resolvedRelationship = $resolvedRelationship->orderBy($sortRule['field'], $sortRule['sortOrder'] ?? "asc");
                                }
                            }

                            return $resolvedRelationship;
                        }
                    });
                }
            }
        }
    }

    protected function resolvePolymorphicFileTagRelationships($endpoint)
    {
        //if exists FileTag model, resolve polymorphic relationship
        if (class_exists('App\\Models\\FileTag')) {
            $modelName = 'App\\Models\\' . Str::singular(Str::studly($endpoint));
            $modelName::resolveRelationUsing('file_tags', function ($model) {
                return $model->morphMany('App\\Models\\FileTag', 'fileable')
                ->orderByRaw('sort_order IS NULL, sort_order')
                ->orderBy('created_at', 'desc');
            });
        }
    }

    protected function getRelationshipLoadArray($endpoint, $loadAllRelationships)
    {
        $schemaRelationships = $this->getSchemaRelationshipsLoadArray($endpoint, $loadAllRelationships);
        $customAttributeRelationships = $this->getCustomAttributesLoadArray($endpoint);
        $loadArray = collect($schemaRelationships)
        ->merge($customAttributeRelationships)
        ->unique()
        ->values()
        ->all();

        return $loadArray;
    }

    protected function getSchemaRelationshipsLoadArray($endpoint, $loadAllRelationships)
    {
        $relationships = new RelationshipBuilder($endpoint, null);
        $hasManyLoadArray = $loadAllRelationships ? $relationships->getHasManyLoadArray() : [];
        $belongsToLoadArray = $relationships->getBelongsToLoadArray();
        $loadArray = collect($hasManyLoadArray)
        ->merge($belongsToLoadArray)
        ->unique()
        ->values()
        ->all();

        return $loadArray;
    }

    protected function getCustomAttributesLoadArray($endpoint)
    {
        $relationships = new RelationshipBuilder($endpoint, null);
        $appendLoadArray = $relationships->getAppendLoadArray();
        $loadArray = collect($appendLoadArray)
        ->unique()
        ->values()
        ->all();

        return $loadArray;
    }

    protected function getSortRules($origin, $relationship)
    {
        $schema = collect(json_decode(File::get(config_path('/pianola/schema.json')), true));
        $tableSchema = collect($schema->where('name', $origin)->first());

        return
        collect($tableSchema['hasMany'] ?? [])->where('name', $relationship)->first()['sortOrder']
        ??
        collect($tableSchema['belongsTo'] ?? [])->where('name', $relationship)->first()['sortOrder']
        ??
        null;
    }

    protected function checkIfModelName($relationship)
    {
        $schema = collect(json_decode(File::get(config_path('/pianola/schema.json')), true));

        return $schema->where('modelName', Str::title($relationship))->count() > 0;
    }

    protected function appendCustomAttributes($endpoint, $records, $loadAllRelationships)
    {
        $schema = json_decode(File::get(config_path('/pianola/schema.json')), true);
        $customAttributes = collect(collect($schema)->where('name', $endpoint)->pluck('custom_attributes')->first())->map(function ($customAttribute) use ($loadAllRelationships) {
            // $loadOnFirstRetrieval = isset($customAttribute['loadOnFirstRetrieval']) ? $customAttribute['loadOnFirstRetrieval'] : false;
            // if ($loadAllRelationships || $loadOnFirstRetrieval) {
            return $customAttribute['name'] ?? null;
            // }
        })->filter()->unique()->toArray();

        return $records->append($customAttributes);
    }

    protected function removeHasManyRelatedRecords($records, $relationships)
    {
        $relationshipsToBeHidden = collect($relationships)->map(function ($relationship) {
            $relationship = explode('.', $relationship)[0];
            if (Str::singular($relationship) == $relationship) {
                return null;
            } else {
                return $relationship;
            }
        })->filter()->unique()->toArray();

        return $records->makeHidden($relationshipsToBeHidden);
    }

    protected function limitResponseSize($array, $relationships, $maxSizeInMb)
    {
        $sizeInBytes = $maxSizeInMb * 1048576;
        $relationships = collect($relationships)->map(function ($relationship) {
            return explode('.', $relationship)[0];
        })->unique()->toArray();
        $size = strlen(json_encode($array));
        if ($size > $sizeInBytes) {
            $sizeMap = [];
            foreach ($relationships as $relationship) {
                $values = collect($array)->pluck($relationship);
                $sizeOfValues = strlen(json_encode($values));
                $sizeMap[] = ['key' => $relationship, 'size' => $sizeOfValues];
            }
            usort($sizeMap, function ($a, $b) {
                return $a['size'] <=> $b['size'];
            });
            $key = array_pop($sizeMap)['key'];
            $array->transform(function ($record) use ($key) {
                unset($record[$key]);

                return $record;
            });
        }

        return $array;
    }

    protected function returnOnlyRequestedHeaders($records, $headers)
    {
        $fields = explode('|', $headers);
        $records = $records->map(function ($record) use ($fields) {
            $returnedRecord = [];
            foreach ($fields as $field) {
                if (strpos($field, '.') !== false) {
                    $tableName = (strtok($field, '.'));
                    $fieldName = str_replace($tableName . '.', '', $field);
                    $returnedRecord[$tableName][$fieldName] = $record[$tableName][$fieldName] ?? null;
                } else {
                    $returnedRecord[$field] = $record[$field];
                }
            }

            return $returnedRecord;
        });

        return $records;
    }
}
