<?php

namespace BalletMecanique\PianolaLaravel\Http\Middleware;

use BalletMecanique\PianolaLaravel\Models\SharedDataUser;
use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;

class SharedDataAuthMiddleware
{
    /**
     * Handle an incoming request.
     *
     * @param  \Closure(\Illuminate\Http\Request): (\Illuminate\Http\Response|\Illuminate\Http\RedirectResponse)  $next
     * @return \Illuminate\Http\Response|\Illuminate\Http\RedirectResponse
     */
    public function handle(Request $request, Closure $next)
    {
        // Check for unlock token first (takes priority over existing authentication)
        $unlockToken = $request->get('unlock_token') ?? $request->header('X-Unlock-Token');

        if ($unlockToken) {
            $configuredUnlockToken = config('pianola-laravel.shared_data_unlock_token');

            // Check if it's the master unlock token (UUID)
            if ($configuredUnlockToken && $unlockToken === $configuredUnlockToken) {
                // Master token grants access - log in as test user
                $testUser = $this->getOrCreateTestUser();

                // Log out any existing sessions
                Auth::guard('shared_data')->logout();

                // Log in the test user
                Auth::guard('shared_data')->login($testUser);

                return $next($request);
            }

            // Otherwise, try to decrypt as an encrypted email
            try {
                // Decrypt the token to get the email
                $email = Crypt::decryptString($unlockToken);

                // Find the user by email
                $user = SharedDataUser::where('email', $email)->first();

                if ($user) {
                    // Log out any existing sessions
                    Auth::guard('shared_data')->logout();
                    $this->logoutExistingSessions($user);

                    // Log in the new user
                    Auth::guard('shared_data')->login($user);

                    return $next($request);
                }
            } catch (\Exception $e) {
                Log::warning('Invalid unlock token provided', [
                    'token' => $unlockToken,
                    'error' => $e->getMessage(),
                ]);
            }
        }

        // Check if user is already authenticated as SharedDataUser (only if no unlock token)
        if (Auth::guard('shared_data')->check()) {
            return $next($request);
        }

        // Neither authenticated nor valid token provided
        // Redirect to login page for web requests
        return redirect()->route('shared-data.login');
    }

    /**
     * Log out existing sessions for the user
     *
     * @return void
     */
    protected function logoutExistingSessions(SharedDataUser $user)
    {
        // Regenerate remember token to invalidate existing sessions
        $user->remember_token = null;
        $user->save();
    }

    /**
     * Get or create a test user for master unlock token access
     */
    protected function getOrCreateTestUser(): SharedDataUser
    {
        $testEmail = 'test@example.com';

        // Try to find existing test user
        $testUser = SharedDataUser::where('email', $testEmail)->first();

        if (! $testUser) {
            // Create test user if it doesn't exist
            $testUser = SharedDataUser::create([
                'name' => 'Test User',
                'email' => $testEmail,
                'password' => Hash::make(Str::random(32)), // Random unguessable password
            ]);

            Log::info('Created test user for master unlock token access', [
                'email' => $testEmail,
            ]);
        }

        return $testUser;
    }
}
