<?php

namespace BalletMecanique\PianolaLaravel\Services;

use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Str;

class QueryBuilder
{
    protected $model;
    protected $queryEncoded;
    protected $sort;
    protected $filter;
    protected $page;
    protected $constantPreFilter;
    protected $queryDecoded;
    protected $orQueries;
    protected $baseTable;

    public function __construct($model, $request)
    {
        $this->model = $model;
        $this->queryEncoded = $request['q'] ?? null;
        $this->constantPreFilter = $request['c'] ?? null;
        $this->variablePreFilter = $request['v'] ?? null;
        $this->sort = $request['sort'] ?? null;
        $this->filter = $request['filter'] ?? null;
        $this->chunk = $request['chunk'] ?? null;
        $this->page = $request['page'] ?? 1;
        $this->baseTable = $this->model->getTable();
        $this->schema = json_decode(File::get(config_path('/pianola/schema.json')), true);
        if ($this->queryEncoded !== null) {
            $this->decodeQuery();
            $this->orQueries = $this->queryDecoded->orQueries;
        }
    }

    protected function decodeQuery()
    {
        $this->queryDecoded = json_decode(
            urldecode(base64_decode($this->queryEncoded))
        );
    }

    public function getQueryJson()
    {
        return urldecode(base64_decode($this->queryEncoded));
    }

    public function get()
    {
        $query = $this->baseQuery();
        //if all pages selected
        if ($this->page === 'all') {
            return $query->get();
        }
        if ($this->chunk) {
            $limit = 1000;
            if ($this->chunk > 1) {
                $offset = ($this->chunk - 1) * $limit;
                $query->offset($offset);
            }

            return $query->limit($limit)->get();
        }
        //if page number selected
        if ($this->page > 1) {
            $offset = ($this->page - 1) * 100;
            $query->offset($offset);
        }

        return $query->limit(100)->get();
    }

    public function count()
    {
        return $this->baseQuery()->distinct($this->baseTable . '.id')->count();
    }

    public function baseQuery()
    {
        $query = $this->model::select($this->baseTable . '.*');
        if ($this->queryDecoded) {
            $this->buildQuery($query);
        }
        if ($this->filter) {
            $this->addFilter($query);
        }
        if ($this->constantPreFilter) {
            $this->addConstantPreFilter($query);
        }
        if ($this->variablePreFilter) {
            $this->addVariablePreFilter($query);
        }
        if ($this->sort) {
            $this->addSort($query);
        }
        // $query->distinct();

        return $query;
    }

    //* this builds the filter

    protected function addFilter($query)
    {
        $filter_columns = collect($this->schema)->where('name', $this->baseTable)->first()['app_record_filter_columns'];
        //filter for concatenation of filter columns
        $query->where(function ($q) use ($filter_columns) {
            $q->whereRaw(
                'concat(' . str_replace([',', '|'], ['," ",', ','], $filter_columns) . ') LIKE ?',
                ['%' . $this->filter . '%']
            );
            //filter for each column individually
            collect(explode(',', $filter_columns))->map(function ($filter_column) use ($q) {
                $q->orWhereRaw('concat(' . str_replace('|', ',', $filter_column) . ') LIKE ?', ['%' . $this->filter . '%']);
            });
        });
    }

    //* this builds the pre-filter

    protected function addConstantPreFilter($query)
    {
        return $this->addFixedFilterQueries($query, $this->constantPreFilter);
    }

    //* this builds the pre-filter

    protected function addVariablePreFilter($query)
    {
        return $this->addFixedFilterQueries($query, $this->variablePreFilter);
    }

    //* this builds the pre-filter
    protected function addFixedFilterQueries($query, $queryString)
    {
        $fieldPath = explode("|", $queryString)[0];
        $value = explode("|", $queryString)[1];

        if (strpos($fieldPath, '.') !== false) {
            $joinTable = explode('.', $fieldPath)[0];
            $field = explode('.', $fieldPath)[1];
            $joinTablePlural = Str::plural($joinTable);
            $baseTableSingular = Str::singular($this->baseTable);
            if ($joinTable == $joinTablePlural) {
                $query->leftJoin($joinTablePlural, $this->baseTable . '.id', '=', $joinTablePlural . '.' . $baseTableSingular . '_id');
            } else {
                $query->leftJoin($joinTablePlural, $this->baseTable . '.' . $joinTable . '_id', '=', $joinTablePlural . '.id');
            }
            $query->where($joinTablePlural . '.' . $field, $value);
        } else {
            $query->where($fieldPath, $value);
        }
    }

    //* this builds the sort

    protected function addSort($query)
    {
        $sort_array = explode(',', $this->sort);
        collect($sort_array)->map(function ($sort) use ($query) {
            $subarray = explode(' ', $sort);
            $field = $subarray[0];
            if (strpos($field, '.') !== false) {
                $tableNameSingular = explode('.', $field)[0];
                $fieldName = explode('.', $field)[1];
                $tableName = Str::plural($tableNameSingular);
                $query->join($tableName, $tableName . '.id', '=', $this->baseTable . '.' . $tableNameSingular . '_id');
            } else {
                $fieldName = $field;
                $tableName = $this->baseTable;
            }
            $asc_desc = $subarray[1] ?? 'asc';

            return $query->orderBy($tableName . '.' . $fieldName, $asc_desc);
        });
    }

    //* this builds the query

    public function buildQuery($query)
    {
        $andQueries = $this->collapsedIncludedAndQueries();
        $this->addJoins($query, $andQueries);
        $this->addConditions($query);
        $query->distinct();

        return $query;
    }

    //* methods used to determine joins

    protected function addJoins($query, $andQueries)
    {
        $this->extractAllJoinInfo($andQueries)->unique()->map(function ($join) use ($query) {
            return $query->leftJoin(
                $join['param_1'],
                $join['param_2'],
                $join['param_3']
            );
        });
    }

    public function extractAllJoinInfo($andQueries)
    {
        return $andQueries
            ->map(function ($andQuery) {
                return $this->extractJoinInfo($andQuery);
            })
            ->filter()
            ->collapse();
    }

    protected function collapsedIncludedAndQueries()
    {
        return $this->collapsedTableGroupings()
            ->map(function ($tableGrouping) {
                if ($tableGrouping->type == 'include') {
                    return $tableGrouping->andQueries;
                }
            })
            ->collapse();
    }

    protected function collapsedTableGroupings()
    {
        return collect($this->orQueries)
            ->map(function ($orQuery) {
                return $orQuery->tableGroupings;
            })
            ->collapse();
    }

    protected function extractJoinInfo($andQuery)
    {
        $queryId = $andQuery->id;
        $field = $andQuery->field;
        $tableString = $this->extractTableStringFromFieldString($field);
        if (strpos($tableString, '.') > -1) {
            //don't create joins for base table!
            $collection = collect(explode('.', $tableString));

            return $collection
                ->map(function ($table, $key) use ($collection, $queryId) {
                    if ($key < $collection->count() - 1) {
                        $leftTableName = $table;
                        $rightTableName = $collection[$key + 1];
                        $array = $collection->toArray();
                        //create a hash-based alias for table on left side of join
                        $leftTableFullString = implode('.', array_slice($array, 0, $key + 1));
                        $aliasForLeftTable = $key > 0 ? $this->createTableAlias($leftTableName, $leftTableFullString, $queryId) : $leftTableName;
                        //create a hash-based alias for table on right side of join
                        $rightTableFullString = implode('.', array_slice($array, 0, $key + 2));
                        $aliasForRightTable = $this->createTableAlias($rightTableName, $rightTableFullString, $queryId);
                        //extract primary and foreign key field names from map
                        $key_fields = $this->lookupRelationshipKeyFields($leftTableName . '.' . $rightTableName);
                        //create parameters from hashed tables and fields
                        $join_parameter_1 = $rightTableName . ' as ' . $aliasForRightTable;
                        $join_parameter_2 = $aliasForLeftTable . '.' . $key_fields[0];
                        $join_parameter_3 = $aliasForRightTable . '.' . $key_fields[1];
                        $result = [
                            'param_1' => $join_parameter_1,
                            'param_2' => $join_parameter_2,
                            'param_3' => $join_parameter_3,
                        ];

                        return collect($result);
                    }
                })
                ->filter();
        }
    }

    //* methods used to determine conditions

    protected function addConditions($query)
    {
        $this->parseOrQueries($query);
    }

    protected function parseOrQueries($query)
    {
        $collection = collect($this->queryDecoded->orQueries);
        $collection->map(function ($orQuery) use ($query) {
            $query->orWhere(function ($q) use ($orQuery) {
                $this->parseTableGroupings($q, $orQuery);
            });
        });
    }

    protected function parseTableGroupings($q, $orQuery)
    {
        $collection = collect($orQuery->tableGroupings);
        $collection->map(function ($tableGrouping) use ($q) {
            if ($tableGrouping->type == 'include') {
                $q->where(function ($q) use ($tableGrouping) {
                    $this->parseAndQueries($q, $tableGrouping);
                });
            } elseif ($tableGrouping->type == 'exclude') {
                $subquery = DB::table($this->baseTable);
                $subquery->distinct($this->baseTable . '.id');
                $subquery->select($this->baseTable . '.id');
                $andQueries = collect($tableGrouping->andQueries);
                $this->addJoins($subquery, $andQueries);
                //add subquery conditions
                $this->parseAndQueries($subquery, $tableGrouping);
                $subquery->get();
                $q->whereNotIn($this->baseTable . '.id', $subquery);
            }
        });
    }

    protected function parseAndQueries($q, $tableGrouping)
    {
        $collection = collect($tableGrouping->andQueries);
        $collection->map(function ($andQuery) use ($q) {
            $field = $this->returnFullAliasedFieldName(
                $andQuery->field,
                $andQuery->id
            );
            $operator = $andQuery->operator;
            $value = $andQuery->value;
            $this->buildQueryStatement($q, $field, $operator, $value);
        });
    }

    private function buildQueryStatement(
        $queryObject,
        $subqueryField,
        $subqueryOperator,
        $subqueryValue
    ) {
        switch ($subqueryOperator) {
            case '=':
                $queryObject->where(
                    $subqueryField,
                    'like',
                    '%' . $subqueryValue . '%'
                );

                break;
            case '==':
                $queryObject->where($subqueryField, '=', $subqueryValue);

                break;
            case '≠':
                $queryObject->where($subqueryField, '<>', $subqueryValue);

                break;
            case '>':
                $queryObject->where($subqueryField, '>', $subqueryValue);

                break;
            case '≥':
                $queryObject->where($subqueryField, '>=', $subqueryValue);

                break;
            case '<':
                $queryObject->where($subqueryField, '<', $subqueryValue);

                break;
            case '≤':
                $queryObject->where($subqueryField, '<=', $subqueryValue);

                break;
            case '*':
                $queryObject
                    ->whereNotNull($subqueryField)
                    ->where($subqueryField, '<>', '');

                break;
            case '∅':
                $queryObject
                    ->where($subqueryField, '=', '')
                    ->orWhereNull($subqueryField);

                break;
            default:
                $queryObject->where(
                    $subqueryField,
                    'like',
                    '%' . $subqueryValue . '%'
                );

                break;
        }
    }

    //alias creation and related methods

    public function returnFullAliasedFieldName($field, $queryId)
    {
        $fullTableString = $this->extractTableStringFromFieldString($field);
        $tableName = $this->extractLastTableNameFromFieldString($field);
        $fieldName = $this->extractFieldNameFromFieldString($field);
        if (strpos($fullTableString, '.') > -1) {
            $alias = $this->createTableAlias(
                $tableName,
                $fullTableString,
                $queryId
            );
        } else {
            $alias = $tableName ?? $this->baseTable;
        }

        return $alias . '.' . $fieldName;
    }

    protected function extractTableStringFromFieldString($field)
    {
        $array = explode('.', $field);
        array_pop($array);
        if (count($array) > 0) {
            return implode('.', $array);
        } else {
            return $this->baseTable;
        }
    }

    protected function extractLastTableNameFromFieldString($field)
    {
        $array = explode('.', $field);
        array_pop($array);
        if (count($array) > 0) {
            return end($array);
        } else {
            return $this->baseTable;
        }
    }

    protected function extractFieldNameFromFieldString($field)
    {
        $array = explode('.', $field);

        return end($array);
    }

    protected function createTableAlias($tableName, $fullTableString, $queryId)
    {
        return $tableName; // . md5($fullTableString . $queryId);
    }

    public function lookupRelationshipKeyFields($relationship)
    {
        $leftTable = explode('.', $relationship)[0];
        $rightTable = explode('.', $relationship)[1];
        $table = collect($this->schema)->where('name', $leftTable)->first();

        $hasMany = collect($table['hasMany'])->map(function ($tables) {
            return $tables['name'];
        })->toArray();

        $belongsTo = collect($table['belongsTo'])->map(function ($tables) {
            return $tables['name'];
        })->toArray();

        if (in_array($rightTable, $hasMany)) {
            $singular = Str::singular($leftTable);

            return ['id', $singular . '_id'];
        } elseif (in_array($rightTable, $belongsTo)) {
            $singular = Str::singular($rightTable);

            return [$singular . '_id', 'id'];
        }
    }
}
