<?php

namespace BalletMecanique\PianolaLaravel\Services;

use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Str;

class SchemaUpdateService
{
    protected $schema;

    public function __construct()
    {
        $this->schema = json_decode(File::get(config_path('/pianola/schema.json')), true);
    }

    public function updateSchema()
    {
        $tables = $this->getDatabaseSchema();

        $schemaCollection = collect($this->schema);

        $newSchema = $this->addMissingColumns($tables, $schemaCollection);
        $newSchema = $this->addMissingModelNames($newSchema);
        $newSchema = $this->addMissingRelationships($newSchema);

        return $newSchema;

        return collect($newSchema)->filter()->values()->all();
    }

    protected function getDatabaseSchema()
    {
        return collect(DB::select('SHOW TABLES'))
        ->map(function ($table) {
            $tableName = reset($table);

            return [
                'tableName' => $tableName,
                'columns' => collect(DB::select('SHOW COLUMNS FROM ' . $tableName))->map(function ($column) {
                    return [
                        'name' => $column->Field,
                        'column_type' => $column->Type,
                        'app_label' => '',
                    ];
                }),
            ];
        });
    }

    protected function addMissingColumns($tables, $schemaCollection)
    {
        $newSchema = [];

        foreach ($tables as $table) {
            $avoidTables = ['password_resets', 'failed_jobs', 'personal_access_tokens'];
            if (
                in_array($table['tableName'], $avoidTables)
                or
                strpos($table['tableName'], 'pianola') !== false
                ) {
                continue;
            }

            //add table if not yet existent
            if (
                $schemaCollection->where('name', $table['tableName'])->count() < 1
            ) {
                $newSchema[] = [
                    'name' => $table['tableName'],
                    'columns' => $table['columns'],
                ];
            } else {
                $subarray = $schemaCollection->where('name', $table['tableName'])->values()->first();
                $newColumnArray = [];
                foreach ($table['columns'] as $column) {
                    if (
                        collect($subarray['columns'])->where('name', $column['name'])->count() < 1
                    ) {
                        $newColumnArray[] = $column;
                    } else {
                        $newColumnArray[] = collect($subarray['columns'])->where('name', $column['name'])->values()->first();
                    }
                }
                $subarray['columns'] = $newColumnArray;
                $newSchema[] = $subarray;
            }
        }

        return collect($newSchema);
    }

    protected function addMissingRelationships($schema)
    {
        $newSchema = [];

        foreach ($schema as $table) {
            //belongsTo
            $belongsToTables = $this->getBelongsToRelationships($table);
            $newBelongsToArray = [];
            foreach ($belongsToTables as $belongsToTable) {
                if (collect($table['belongsTo'] ?? [])->where('name', $belongsToTable)->count() < 1) {
                    $newBelongsToArray[] = ['name' => $belongsToTable];
                } else {
                    $newBelongsToArray[] = collect($table['belongsTo'])->where('name', $belongsToTable)->values()->first();
                }
            }
            $table['belongsTo'] = $newBelongsToArray;

            //hasMany
            $hasManyTables = $this->getHasManyRelationships($table, $schema);
            $newHasManyArray = [];
            foreach ($hasManyTables as $hasManyTable) {
                if (collect($table['hasMany'] ?? [])->where('name', $hasManyTable)->count() < 1) {
                    $newHasManyArray[] = ['name' => $hasManyTable];
                } else {
                    $newHasManyArray[] = collect($table['hasMany'])->where('name', $hasManyTable)->values()->first();
                }
            }
            $table['hasMany'] = $newHasManyArray;
            $newSchema[] = $table;
        };

        return $newSchema;
    }

    protected function addMissingModelNames($schema)
    {
        $newSchema = [];
        foreach ($schema as $table) {
            $modelName = Str::singular(Str::studly($table['name']));
            $table['modelName'] = $table['modelName'] ?? $modelName;
            $newSchema[] = $table;
        }

        return $newSchema;
    }

    protected function getBelongsToRelationships($table)
    {
        return collect($table['columns'])->map(function ($column) {
            if (Str::endsWith($column['name'], '_id')) {
                $modelName = Str::studly(Str::replace('_id', '', $column['name']));
                if ($this->getTableNameFromModelName($modelName)) {
                    return $this->getTableNameFromModelName($modelName);
                }
            }
        })->filter()->values();
    }

    protected function getHasManyRelationships($table, $schema)
    {
        $foreignKeyFieldName = Str::singular($table['name']) . '_id';

        return collect($schema)->map(function ($table) use ($foreignKeyFieldName) {
            if (collect($table['columns'])->where('name', $foreignKeyFieldName)->count() > 0) {
                if ($this->tableNameExistsInSchema($table['name'])) {
                    return $table['name'];
                }
            };
        })->filter()->values();
    }

    public function createModels()
    {
        $models = collect($this->schema)->pluck('modelName');
        $models->each(function ($model) {
            $modelPath = 'Models/' . $model . '.php';
            if (! file_exists(app_path($modelPath))) {
                $this->createFile(app_path($modelPath), $this->getModelString($model));
            };
        });

        return 'ok';
    }

    protected function tableNameExistsInSchema($tableName)
    {
        return collect($this->schema)->where('name', $tableName)->count() > 0;
    }

    protected function createFile($filepath, $content)
    {
        if (! file_exists($filepath)) {
            File::put($filepath, $content);
        }
    }

    protected function getModelString($model)
    {
        $tableName = $this->getTableNameFromModelName($model);
        $pluralisedModelName = Str::snake(Str::plural($model));
        $showModelString = $tableName !== $pluralisedModelName;
        $tableString = $showModelString ? ("\n".'    protected $table = "' . $tableName . '";') : "";

        return <<<EOT
        <?php

        namespace App\Models;

        use Illuminate\Database\Eloquent\Factories\HasFactory;
        use Illuminate\Database\Eloquent\Model;

        class $model extends Model
        {
            use HasFactory;

            protected \$guarded = [];$tableString

        }

        EOT;
    }

    protected function getTableNameFromModelName($modelName)
    {
        return collect($this->schema)->where('modelName', $modelName)->first()['name'] ?? '';
    }
}
