<?php

namespace BalletMecanique\PianolaLaravel\Http\Controllers;

use BalletMecanique\PianolaLaravel\Models\PianolaCalendarPreset;
use BalletMecanique\PianolaLaravel\Services\QueryBuilder;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Str;

class CalendarPrintoutController
{
    public function __invoke($endpoint, Request $request)
    {
        $type = $request->input('type');
        $date = $request->input('date');
        $calPage = $request->input('calPage');

        // Check if this is a print all pages request
        if ($calPage === 'all') {
            return $this->printAllPages($endpoint, $request, $type);
        }

        $events = $this->getEvents($endpoint, $request, $type, $date);
        $locations = $this->getLocations($endpoint, $request);
        $logoPath = $this->getLogoPath();
        $title = $this->getTitle($date, $type);

        return view('pianola-laravel::printouts.calendar', compact('events', 'date', 'locations', 'title', 'logoPath', 'type'));
    }

    protected function getLocationForeignKeyFieldName($endpoint)
    {
        $endpointName = $this->getModuleConfigParameter($endpoint, 'calendarLocationEndpoint');
        return Str::singular(Str::snake($endpointName)) . '_id';
    }

    protected function getEvents($endpoint, $request, $type, $date)
    {
        $modelName = 'App\\Models\\'.Str::singular(Str::studly($endpoint));
        $model = new $modelName;
        $query = new QueryBuilder($model, $request);
        $query = $query->baseQuery();
        if ($type === 'daily') {
            $query->where('date', $date);
        } else {
            $mondayBefore = date('Y-m-d', strtotime($date.' - '.date('w', strtotime($date)) - 1 .' days'));
            $query->where('date', '>=', date('Y-m-d', strtotime($mondayBefore)))
                ->where('date', '<=', date('Y-m-d', strtotime($mondayBefore.' + 6 days')))
                ->whereIn($this->getLocationForeignKeyFieldName($endpoint), $this->getLocationIds($endpoint, $request));
        }
        $records = $query->get();
        $records = $this->returnOnlyCalendarData($records, $endpoint);

        return $records->values();
    }

    protected function getLocationIds($endpoint, $request)
    {
        $preset = PianolaCalendarPreset::find($request->input('preset'));
        $presetData = json_decode($preset->json, false, 512, JSON_THROW_ON_ERROR);

        return collect($presetData)->where('number', $request->input('calPage'))->pluck('location_ids')->first();
    }

    protected function returnOnlyCalendarData($records, $endpoint)
    {
        $schema = collect(json_decode(File::get(config_path('/pianola/schema.json')), true));
        $endpointSchema = $schema->where('name', $endpoint)->first();
        $calendarMapping = $endpointSchema['appCalendarMapping'] ?? [];
        $map = [
            'id' => $calendarMapping['id'] ?? 'id',
            'date' => $calendarMapping['date'] ?? 'date',
            'timestamp_start' => $calendarMapping['timestamp_start'] ?? 'timestamp_start',
            'timestamp_end' => $calendarMapping['timestamp_end'] ?? 'timestamp_end',
            'location' => $calendarMapping['location'] ?? 'location_name',
            'name' => $calendarMapping['name'] ?? 'calendar_event_name',
            'more_information' => $calendarMapping['more_information'] ?? 'calendar_information',
            'locked' => $calendarMapping['locked'] ?? 'calendar_locked',
            'color' => $calendarMapping['color'] ?? 'calendar_color',
        ];
        $customAttributes = $endpointSchema['custom_attributes'] ?? [];
        $relationshipDependencies = collect($customAttributes)->filter(function ($attribute) use ($map) {
            return in_array($attribute['name'], $map);
        })->pluck('relationshipDependencies')->flatten();

        $records->load($relationshipDependencies->toArray());

        return $records->map(function ($record) use ($map) {
            if (empty($record[$map['date']])) {
                return null;
            }
            if (empty($record[$map['timestamp_start']])) {
                return null;
            }
            if (empty($record[$map['timestamp_end']])) {
                $record[$map['timestamp_end']] = date('Y-m-d H:i:s', strtotime($record[$map['timestamp_start']]) + 900);
            }

            // Truncate timestamp_end if it goes beyond midnight
            $eventDate = $record[$map['date']];
            $timestampEnd = $record[$map['timestamp_end']];
            $endDate = date('Y-m-d', strtotime($timestampEnd));

            // If the end timestamp is on a different date than the event date, truncate to 23:59 of the event date
            if ($endDate !== $eventDate) {
                $timestampEnd = $eventDate . ' 23:59:00';
            } else {
                $timestampEnd = date('Y-m-d H:i:s', strtotime($timestampEnd));
            }

            return [
                'date' => $record[$map['date']],
                'calendar_id' => $record[$map['id']],
                'calendar_timestamp_start' => $record[$map['timestamp_start']],
                'calendar_timestamp_end' => $timestampEnd,
                'calendar_location' => $record[$map['location']],
                'calendar_name' => $record[$map['name']],
                'calendar_more_information' => $record[$map['more_information']],
                'calendar_locked' => $record[$map['locked']],
                'calendar_color' => $this->correctColor($record[$map['color']]),
            ];
        })->filter()->values();
    }

    protected function correctColor($color)
    {
        if (empty($color)) {
            return '#ccc';
        }
        if (strpos($color, '#') === false) {
            return '#'.$color;
        }

        return $color;
    }

    protected function getModuleConfigParameter($endpoint, $parameter)
    {
        $appJson = file_get_contents(config_path('pianola/app.json'));
        $app = json_decode($appJson, true, 512, JSON_THROW_ON_ERROR);
        $module = collect($app['modules'])->where('basicConfig.apiEndpoint', $endpoint)->first();
        return $module['basicConfig'][$parameter] ?? null;
    }

    protected function getLocations($endpoint, $request)
    {
        $preset = PianolaCalendarPreset::find($request->input('preset'));
        $presetData = json_decode($preset->json, false, 512, JSON_THROW_ON_ERROR);
        $locationIds = collect($presetData)->where('number', $request->input('calPage'))->pluck('location_ids')->first();

        $calendarLocationNameColumn = $this->getModuleConfigParameter($endpoint, 'calendarLocationNameColumn') ?? 'name';
        $calendarLocationEndpoint = $this->getModuleConfigParameter($endpoint, 'calendarLocationEndpoint') ?? 'locations';
        $locationModelName = Str::singular(Str::studly($calendarLocationEndpoint));

        return collect($locationIds)->map(function ($locationId) use ($calendarLocationNameColumn, $locationModelName) {
            $modelName = 'App\\Models\\'. $locationModelName;
            $model = new $modelName;

            return $model::find($locationId)->{$calendarLocationNameColumn};
        })->values();
    }

    protected function getLogoPath()
    {
        $appJson = file_get_contents(config_path('pianola/app.json'));
        $app = json_decode($appJson, true, 512, JSON_THROW_ON_ERROR);

        return $app['basicConfig']['logoPath'];
    }

    protected function getTitle($date, $type)
    {
        $date = date('d.m.Y', strtotime($date));
        $weekday = match (date('l', strtotime($date))) {
            'Monday' => 'Montag',
            'Tuesday' => 'Dienstag',
            'Wednesday' => 'Mittwoch',
            'Thursday' => 'Donnerstag',
            'Friday' => 'Freitag',
            'Saturday' => 'Samstag',
            'Sunday' => 'Sonntag',
        };
        if ($type === 'daily') {
            return 'Tagesübersicht '.$weekday.', '.$date;
        }
        $mondayBefore = date('Y-m-d', strtotime($date.' - '.date('w', strtotime($date)) - 1 .' days'));

        return 'Wochenübersicht: '.date('d.m.Y', strtotime($mondayBefore)).' - '.date('d.m.Y', strtotime($mondayBefore.' + 6 days'));
    }

    /**
     * Handle printing all pages/periods
     */
    protected function printAllPages($endpoint, $request, $type)
    {
        // Get all events without date filtering
        $allEvents = $this->getAllEvents($endpoint, $request);

        // Get all periods data for printing
        $allPeriodsData = $this->getAllPeriodsForPrint($allEvents, $type, $endpoint, $request);

        // Get app configuration
        $logoPath = $this->getLogoPath();
        $listName = $this->getListName($endpoint);
        $appConfig = $this->getAppConfig();

        // Set app locale for translations
        app()->setLocale($appConfig['appLanguage'] ?? 'en');

        return view('pianola-laravel::printouts.calendar-all', compact('allPeriodsData', 'logoPath', 'type', 'listName', 'appConfig'));
    }

    /**
     * Get all events without date filtering
     */
    protected function getAllEvents($endpoint, $request)
    {
        $modelName = 'App\\Models\\'.Str::singular(Str::studly($endpoint));
        $model = new $modelName;
        $query = new QueryBuilder($model, $request);
        $query = $query->baseQuery();

        // Apply location filtering if calPage is specified and not 'all'
        $calPage = $request->input('calPage');
        if ($calPage && $calPage !== 'all') {
            $query->whereIn($this->getLocationForeignKeyFieldName($endpoint), $this->getLocationIds($endpoint, $request));
        }

        $records = $query->get();
        $records = $this->returnOnlyCalendarData($records, $endpoint);

        return $records->values();
    }

    /**
     * Get all periods data for printing (all days or all weeks)
     */
    protected function getAllPeriodsForPrint($calendarData, $type, $endpoint, $request)
    {
        $allPeriods = [];

        if ($type === 'daily') {
            // Get all unique dates
            $dates = collect($calendarData)->pluck('date')->unique()->sort()->values();

            foreach ($dates as $date) {
                $events = collect($calendarData)->filter(function ($event) use ($date) {
                    return $event['date'] === $date;
                })->values()->toArray();

                if (! empty($events)) {
                    $locations = $this->getLocationsForPrint($events, $endpoint, $request);
                    $title = $this->getTitle($date, 'daily');

                    $allPeriods[] = [
                        'date' => $date,
                        'events' => $events,
                        'locations' => $locations,
                        'title' => $title,
                        'type' => 'daily',
                    ];
                }
            }
        } else {
            // Weekly view - get all unique weeks
            $dates = collect($calendarData)->pluck('date')->unique()->sort()->values();
            $weeks = [];

            foreach ($dates as $date) {
                $mondayBefore = date('Y-m-d', strtotime($date.' - '.date('w', strtotime($date)) - 1 .' days'));
                if (! in_array($mondayBefore, $weeks)) {
                    $weeks[] = $mondayBefore;
                }
            }

            foreach ($weeks as $weekStart) {
                $weekEnd = date('Y-m-d', strtotime($weekStart.' + 6 days'));

                $events = collect($calendarData)->filter(function ($event) use ($weekStart, $weekEnd) {
                    return $event['date'] >= $weekStart && $event['date'] <= $weekEnd;
                })->values()->toArray();

                if (! empty($events)) {
                    $locations = $this->getLocationsForPrint($events, $endpoint, $request);
                    $title = $this->getTitle($weekStart, 'weekly');

                    $allPeriods[] = [
                        'date' => $weekStart,
                        'events' => $events,
                        'locations' => $locations,
                        'title' => $title,
                        'type' => 'weekly',
                    ];
                }
            }
        }

        return $allPeriods;
    }

    /**
     * Get unique locations from filtered events for print all functionality
     * Maintains the same order as defined in the calendar preset
     */
    protected function getLocationsForPrint($events, $endpoint, $request)
    {
        // Get the predefined location order from the preset
        $preset = PianolaCalendarPreset::find($request->input('preset'));
        $presetData = json_decode($preset->json, false, 512, JSON_THROW_ON_ERROR);

        // Get all location IDs from all calendar pages in the preset
        $allLocationIds = collect($presetData)->pluck('location_ids')->flatten()->unique()->filter();

        $appJson = file_get_contents(config_path('pianola/app.json'));
        $app = json_decode($appJson, true, 512, JSON_THROW_ON_ERROR);
        $module = collect($app['modules'])->where('basicConfig.apiEndpoint', $endpoint)->first();
        $calendarLocationNameColumn = $module['basicConfig']['calendarLocationNameColumn'] ?? 'name';

        // Get location names in the preset order
        $orderedLocationNames = $allLocationIds->map(function ($locationId) use ($calendarLocationNameColumn) {
            $modelName = 'App\\Models\\Location';
            $model = new $modelName;
            $location = $model::find($locationId);
            return $location ? $location->{$calendarLocationNameColumn} : null;
        })->filter();

        // Get unique locations that actually appear in the events
        $eventLocationNames = collect($events)->pluck('calendar_location')->filter()->unique();

        // Return locations in preset order, but only include those that have events
        return $orderedLocationNames->filter(function ($locationName) use ($eventLocationNames) {
            return $eventLocationNames->contains($locationName);
        })->values()->toArray();
    }

    /**
     * Get list name for the endpoint
     */
    protected function getListName($endpoint)
    {
        $appJson = file_get_contents(config_path('pianola/app.json'));
        $app = json_decode($appJson, true, 512, JSON_THROW_ON_ERROR);

        // Find module where basicConfig.apiEndpoint === $endpoint
        $module = collect($app['modules'])->where('basicConfig.apiEndpoint', $endpoint)->first();

        return $module['basicConfig']['name'] ?? ucfirst($endpoint);
    }

    /**
     * Get app configuration
     */
    protected function getAppConfig()
    {
        $appJson = file_get_contents(config_path('pianola/app.json'));
        $app = json_decode($appJson, true, 512, JSON_THROW_ON_ERROR);

        return [
            'appLanguage' => $app['basicConfig']['appLanguage'] ?? 'en',
            'logoPath' => $app['basicConfig']['logoPath'] ?? null,
        ];
    }
}
