<?php

namespace BalletMecanique\PianolaLaravel\Services;

use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Str;

class QueryBuilder
{
    protected $model;

    protected $queryEncoded;

    protected $sort;

    protected $filter;

    protected $page;

    protected $constantPreFilter;

    protected $queryDecoded;

    protected $orQueries;

    protected $baseTable;

    protected $schema;

    protected $chunk;

    protected $variablePreFilter;

    public function __construct($model, $request)
    {
        $this->model = $model;
        $this->queryEncoded = $request['q'] ?? null;
        $this->constantPreFilter = $request['c'] ?? null;
        $this->variablePreFilter = $request['v'] ?? null;
        $this->sort = $request['sort'] ?? null;
        $this->filter = $request['filter'] ?? null;
        $this->chunk = $request['chunk'] ?? null;
        $this->page = $request['page'] ?? 1;
        $this->baseTable = $this->model->getTable();
        $this->schema = json_decode(File::get(config_path('/pianola/schema.json')), true);
        $this->saveQueryParametersToSession();
        if ($this->queryEncoded !== null) {
            $this->decodeQuery();
            $this->orQueries = $this->queryDecoded->orQueries;
        }
    }

    protected function decodeQuery()
    {
        $this->queryDecoded = json_decode(
            urldecode(base64_decode($this->queryEncoded))
        );
    }

    protected function saveQueryParametersToSession()
    {
        session()->put($this->baseTable.'.query', $this->queryEncoded);
        session()->put($this->baseTable.'.sort', $this->sort);
        session()->put($this->baseTable.'.filter', $this->filter);
        session()->put($this->baseTable.'.page', $this->page);
    }

    public function get()
    {
        $query = $this->baseQuery();
        //if all pages selected
        if ($this->page === 'all') {
            return $query->get();
        }
        if ($this->chunk) {
            $limit = 1000;
            if ($this->chunk > 1) {
                $offset = ($this->chunk - 1) * $limit;
                $query->offset($offset);
            }

            return $query->limit($limit)->get();
        }
        //if page number selected
        if ($this->page > 1) {
            $offset = ($this->page - 1) * 100;
            $query->offset($offset);
        }

        return $query->limit(100)->get();
    }

    public function count()
    {
        return $this->baseQuery()->distinct($this->baseTable.'.id')->count();
    }

    public function baseQuery()
    {
        $query = $this->model::select($this->baseTable.'.*', ...$this->getSortStringArray());
        if ($this->queryDecoded) {
            $this->buildQuery($query);
        }
        if ($this->filter) {
            $this->addFilter($query);
        }
        if ($this->constantPreFilter) {
            $this->addConstantPreFilter($query);
        }
        if ($this->variablePreFilter) {
            $this->addVariablePreFilter($query);
        }
        if ($this->sort) {
            $this->addSort($query);
        }
        // $query->distinct();

        return $query;
    }

    //* this builds the filter

    protected function addFilter($query)
    {
        $filterValue = $this->getStringWithReplacedWildcardCharactersForLikeSearch($this->filter);
        $filterValue = $this->replaceDateNotation($filterValue);
        $filter_columns = collect($this->schema)->where('name', $this->baseTable)->first()['app_record_filter_columns'];
        //if any of the filter columns includes a dot notation, create a join
        $joinedTables = [];
        collect(explode(',', $filter_columns))->map(function ($filter_column) use ($query, &$joinedTables) {
            if (strpos($filter_column, '.') !== false) {
                $baseTable = $this->baseTable;
                $joinTable = trim(explode('.', $filter_column)[0]);
                // Check if the table has already been joined
                if (! in_array($joinTable, $joinedTables)) {
                    $aliasedJoinTable = 'filterJoin_'.$joinTable;
                    $key_fields = $this->lookupRelationshipKeyFields($baseTable.'.'.$joinTable);
                    $query->leftJoin($joinTable.' as '.$aliasedJoinTable, $baseTable.'.'.$key_fields[0], '=', $aliasedJoinTable.'.'.$key_fields[1]);
                    // Add the joined table to the list
                    $joinedTables[] = $joinTable;
                }
            }
        });
        //map filter_columns: if any column contains a dot, replace it with the aliased join table
        $filter_columns = collect(explode(',', $filter_columns))->map(function ($filter_column) {
            if (strpos($filter_column, '.') !== false) {
                $aliasedJoinTable = 'filterJoin_'.trim(explode('.', $filter_column)[0]);

                return $aliasedJoinTable.'.'.trim(explode('.', $filter_column)[1]);
            }

            return $this->baseTable.'.'.$filter_column;
        })->implode(',');
        //filter for concatenation of filter columns
        $query->where(function ($q) use ($filter_columns, $filterValue) {
            $q->whereRaw(
                'concat('.str_replace(',', '," ",', $filter_columns).') LIKE ?',
                ['%'.$filterValue.'%']
            );
            //filter for each column individually
            collect(explode(',', $filter_columns))->map(function ($filter_column) use ($q, $filterValue) {
                $q->orWhereRaw('concat('.$filter_column.') LIKE ?', ['%'.$filterValue.'%']);
            });
            //query each space-divided term in all columns
            $terms = explode(' ', $this->filter);
            $filter_columns_array = explode(',', $filter_columns);
            $q->orWhere(function ($qu) use ($terms, $filter_columns_array) {
                foreach ($terms as $term) {
                    $qu->where(function ($q) use ($filter_columns_array, $term) {
                        foreach ($filter_columns_array as $filter_column) {
                            $q->orWhere($filter_column, 'LIKE', '%'.$term.'%');
                        }
                    });
                }
            });
        });
    }

    //* this builds the pre-filter

    protected function addConstantPreFilter($query)
    {
        $this->addFixedFilterQueries($query, $this->constantPreFilter);
    }

    //* this builds the pre-filter

    protected function addVariablePreFilter($query)
    {
        $this->addFixedFilterQueries($query, $this->variablePreFilter);
    }

    //* this builds the pre-filter
    protected function addFixedFilterQueries($query, $queryString)
    {
        $fieldPath = explode('|', $queryString)[0];
        $value = explode('|', $queryString)[1];
        $anyValue = '*';
        if (strpos($fieldPath, '.') !== false) {
            $joinTable = explode('.', $fieldPath)[0];
            $field = explode('.', $fieldPath)[1];
            $joinTablePlural = Str::plural($joinTable);
            $baseTableSingular = Str::singular($this->baseTable);
            if ($joinTable == $joinTablePlural) {
                $query->leftJoin($joinTablePlural, $this->baseTable.'.id', '=', $joinTablePlural.'.'.$baseTableSingular.'_id');
            } else {
                $query->leftJoin($joinTablePlural, $this->baseTable.'.'.$joinTable.'_id', '=', $joinTablePlural.'.id');
            }
            if ($value === $anyValue) {
                $query->whereNotNull($joinTablePlural.'.'.$field)
                    ->whereNot($joinTablePlural.'.'.$field, '');
            } else {
                $query->where($joinTablePlural.'.'.$field, $value);
            }
        } else {
            if ($value === $anyValue) {
                $query->whereNotNull($this->baseTable.'.'.$fieldPath)
                    ->whereNot($this->baseTable.'.'.$fieldPath, '');
            } else {
                $query->where($this->baseTable.'.'.$fieldPath, $value);
            }
        }
    }

    //* this builds the sort

    protected function addSort($query)
    {
        $sort_array = explode(',', $this->sort);
        $joinedTables = collect();  // Collection to keep track of joined tables

        // First step: Process joins
        collect($sort_array)->each(function ($sort) use ($query, $joinedTables) {
            $subarray = explode(' ', $sort);
            $field = $subarray[0];

            if (strpos($field, '.0.') !== false) {
                $relatedTableName = explode('.0.', $field)[0];
                $baseTableNameSingular = Str::singular($this->baseTable);
                $tableName = $relatedTableName;

                if (! $joinedTables->contains($tableName)) {
                    $query->join($tableName, $this->baseTable.'.id', '=', $relatedTableName.'.'.$baseTableNameSingular.'_id');
                    $joinedTables->push($tableName);  // Mark table as joined
                }
            } elseif (strpos($field, '.') !== false) {
                $tableNameSingular = explode('.', $field)[0];
                $tableName = Str::plural($tableNameSingular);

                if (! $joinedTables->contains($tableName)) {
                    $query->join($tableName, $tableName.'.id', '=', $this->baseTable.'.'.$tableNameSingular.'_id');
                    $joinedTables->push($tableName);  // Mark table as joined
                }
            }
        });

        // Second step: Process orderBy
        collect($sort_array)->each(function ($sort) use ($query) {
            $subarray = explode(' ', $sort);
            $field = $subarray[0];
            $asc_desc = $subarray[1] ?? 'asc';

            if (strpos($field, '.0.') !== false) {
                $relatedTableName = explode('.0.', $field)[0];
                $fieldName = explode('.0.', $field)[1];
                $tableName = $relatedTableName;
            } elseif (strpos($field, '.') !== false) {
                $tableNameSingular = explode('.', $field)[0];
                $fieldName = explode('.', $field)[1];
                $tableName = Str::plural($tableNameSingular);
            } else {
                $fieldName = $field;
                $tableName = $this->baseTable;
            }

            $query->orderBy($tableName.'.'.$fieldName, $asc_desc);
        });
    }

    protected function getSortStringArray()
    {
        if (! $this->sort) {
            return collect([]);
        }
        $sort_array = explode(',', $this->sort);

        return collect($sort_array)->map(function ($sort) {
            $subarray = explode(' ', $sort);
            $field = $subarray[0];
            if (strpos($field, '.0.') !== false) {
                $relatedTable = explode('.0.', $field)[0];
                $fieldName = explode('.0.', $field)[1];
                $tableName = $relatedTable;
            } elseif (strpos($field, '.') !== false) {
                $tableNameSingular = explode('.', $field)[0];
                $fieldName = explode('.', $field)[1];
                $tableName = Str::plural($tableNameSingular);
            } else {
                $fieldName = $field;
                $tableName = $this->baseTable;
            }

            return $tableName.'.'.$fieldName.' as '.$tableName.'_'.$fieldName;
        });
    }

    //* this builds the query

    public function buildQuery($query)
    {
        $query->where(function ($query) {
            collect($this->orQueries)->each(function ($orQuery) use ($query) {
                if (collect($orQuery->tableGroupings)->count() === 0) {
                    return;
                }
                $query->orWhereIn($this->baseTable.'.id', function ($subquery) use ($orQuery) {
                    $subquery->select($this->baseTable.'.id');
                    $subquery->from($this->baseTable);
                    $this->addJoinsForOrQuery($subquery, $orQuery);
                    $this->parseTableGroupings($subquery, $orQuery);
                });
            });
        });
    }

    protected function addJoinsForOrQuery($query, $orQuery)
    {
        $tableGroupings = $this->getTableGroupings($orQuery);
        $andQueries = $this->getCollapsedAndQueries($tableGroupings->where('type', 'include'));
        $this->addJoins($query, $andQueries);
    }

    protected function getTableGroupings($orQuery)
    {
        return collect($orQuery->tableGroupings);
    }

    //* methods used to determine joins

    protected function addJoins($query, $andQueries)
    {
        $this->extractAllJoinInfo($andQueries)->unique()->map(function ($join) use ($query) {
            return $query->leftJoin(
                $join['param_1'],
                $join['param_2'],
                $join['param_3']
            );
        });
    }

    public function extractAllJoinInfo($andQueries)
    {
        return $andQueries
            ->map(function ($andQuery) {
                return $this->extractJoinInfo($andQuery);
            })
            ->filter()
            ->collapse();
    }

    protected function getCollapsedAndQueries($tableGroupings)
    {
        return $tableGroupings->map(function ($tableGrouping) {
            $andQueries = $tableGrouping->andQueries;

            return collect($andQueries)->map(function ($andQuery) use ($tableGrouping) {
                $andQuery->tableGroupingId = $tableGrouping->id;

                return $andQuery;
            });
        })
            ->collapse();
    }

    protected function extractJoinInfo($andQuery)
    {
        $tableGroupingId = $andQuery->tableGroupingId;
        $field = $andQuery->field;
        $tableString = $this->extractTableStringFromFieldString($field);
        if (strpos($tableString, '.') > -1) {
            //don't create joins for base table!
            $collection = collect(explode('.', $tableString));

            return $collection
                ->map(function ($table, $key) use ($collection, $tableGroupingId) {
                    if ($key < $collection->count() - 1) {
                        $leftTableName = $table;
                        $rightTableName = $collection[$key + 1];
                        //create a hash-based alias for table on left side of join
                        $aliasForLeftTable = $key > 0 ? $this->createTableAlias($leftTableName, $tableGroupingId) : $leftTableName;
                        //create a hash-based alias for table on right side of join
                        $aliasForRightTable = $this->createTableAlias($rightTableName, $tableGroupingId);
                        //extract primary and foreign key field names from map
                        $key_fields = $this->lookupRelationshipKeyFields($leftTableName.'.'.$rightTableName);
                        //create parameters from hashed tables and fields
                        $join_parameter_1 = $rightTableName.' as '.$aliasForRightTable;
                        $join_parameter_2 = $aliasForLeftTable.'.'.$key_fields[0];
                        $join_parameter_3 = $aliasForRightTable.'.'.$key_fields[1];
                        $result = [
                            'param_1' => $join_parameter_1,
                            'param_2' => $join_parameter_2,
                            'param_3' => $join_parameter_3,
                        ];

                        return collect($result);
                    }

                    return null;
                })
                ->filter();
        }

        return null;
    }

    protected function parseTableGroupings($q, $orQuery)
    {
        $collection = collect($orQuery->tableGroupings);
        $collection->map(function ($tableGrouping) use ($q) {
            if ($tableGrouping->type === 'include') {
                $q->where(function ($q) use ($tableGrouping) {
                    $this->parseAndQueries($q, $tableGrouping);
                });
            } elseif ($tableGrouping->type === 'exclude') {
                $subquery = DB::table($this->baseTable);
                $subquery->distinct($this->baseTable.'.id');
                $subquery->select($this->baseTable.'.id');
                $andQueries = collect($tableGrouping->andQueries)
                    ->map(function ($andQuery) use ($tableGrouping) {
                        $andQuery->tableGroupingId = $tableGrouping->id;

                        return $andQuery;
                    });
                $this->addJoins($subquery, $andQueries);
                //add subquery conditions
                $this->parseAndQueries($subquery, $tableGrouping);
                $subquery->get();
                $q->whereNotIn($this->baseTable.'.id', $subquery);
            }
        });
    }

    protected function parseAndQueries($q, $tableGrouping)
    {
        $collection = collect($tableGrouping->andQueries);
        $tableGroupingId = $tableGrouping->id;
        $collection->map(function ($andQuery) use ($q, $tableGroupingId) {
            $field = $this->returnFullAliasedFieldName(
                $andQuery->field,
                $tableGroupingId
            );
            $operator = $andQuery->operator;
            $value = $andQuery->value;
            $this->buildQueryStatement($q, $field, $operator, $value);
        });
    }

    private function buildQueryStatement(
        $queryObject,
        $subqueryField,
        $subqueryOperator,
        $subqueryValue
    ) {
        switch ($subqueryOperator) {
            case '=':
                $queryObject->where(
                    $subqueryField,
                    'like',
                    '%'.$this->getStringWithReplacedWildcardCharactersForLikeSearch($subqueryValue).'%'
                );

                break;
            case '==':
                if ($subqueryValue == false) {
                    $queryObject->where(function ($q) use ($subqueryField) {
                        $q->whereNull($subqueryField)
                            ->orWhere($subqueryField, '')
                            ->orWhere($subqueryField, 'false')
                            ->orWhere($subqueryField, '0');
                    });
                } else {
                    $queryObject->where($subqueryField, '=', $subqueryValue);
                }

                break;
            case '≠':
                $queryObject->where(function ($q) use ($subqueryField, $subqueryValue) {
                    $q->where($subqueryField, '<>', $subqueryValue)
                        ->orWhereNull($subqueryField)
                        ->orWhere($subqueryField, '=', '');
                });

                break;
            case '>':
                $queryObject->where($subqueryField, '>', $subqueryValue);

                break;
            case '>=':
                $queryObject->where($subqueryField, '>=', $subqueryValue);

                break;
            case '<':
                $queryObject->where($subqueryField, '<', $subqueryValue);

                break;
            case '<=':
                $queryObject->where($subqueryField, '<=', $subqueryValue);

                break;
            case '*':
                $queryObject
                    ->whereNotNull($subqueryField)
                    ->where($subqueryField, '<>', '');

                break;
            case '∅':
                $queryObject
                    ->where($subqueryField, '=', '')
                    ->orWhereNull($subqueryField);

                break;
            default:
                $queryObject->where(
                    $subqueryField,
                    'like',
                    '%'.$this->getStringWithReplacedWildcardCharactersForLikeSearch($subqueryValue).'%'
                );

                break;
        }
    }

    //alias creation and related methods

    public function returnFullAliasedFieldName($field, $tableGroupingId)
    {
        $fullTableString = $this->extractTableStringFromFieldString($field);
        $tableName = $this->extractLastTableNameFromFieldString($field);
        $fieldName = $this->extractFieldNameFromFieldString($field);
        if (strpos($fullTableString, '.') > -1) {
            $alias = $this->createTableAlias(
                $tableName,
                $tableGroupingId
            );
        } else {
            $alias = $tableName ?? $this->baseTable;
        }

        return $alias.'.'.$fieldName;
    }

    protected function extractTableStringFromFieldString($field)
    {
        $array = explode('.', $field);
        array_pop($array);
        if (count($array) > 0) {
            return implode('.', $array);
        }

        return $this->baseTable;
    }

    protected function extractLastTableNameFromFieldString($field)
    {
        $array = explode('.', $field);
        array_pop($array);
        if (count($array) > 0) {
            return end($array);
        }

        return $this->baseTable;
    }

    protected function extractFieldNameFromFieldString($field)
    {
        $array = explode('.', $field);

        return end($array);
    }

    protected function createTableAlias($tableName, $tableGroupingId)
    {
        return str_replace('.', '_', $tableName.'_'.$tableGroupingId);
    }

    public function lookupRelationshipKeyFields($relationship)
    {
        $leftTable = explode('.', $relationship)[0];
        $rightTable = explode('.', $relationship)[1];
        $table = collect($this->schema)->where('name', $leftTable)->first();

        $hasMany = collect($table['hasMany'])->map(function ($tables) {
            return $tables['name'];
        })->toArray();

        $belongsTo = collect($table['belongsTo'])->map(function ($tables) {
            return $tables['name'];
        })->toArray();

        if (in_array($rightTable, $hasMany)) {
            $singular = Str::singular($leftTable);

            return ['id', $singular.'_id'];
        }

        if (in_array($rightTable, $belongsTo)) {
            $singular = Str::singular($rightTable);

            return [$singular.'_id', 'id'];
        }

        return null;
    }

    public function getStringWithReplacedWildcardCharactersForLikeSearch($input)
    {
        return str_replace(['\'', '’'], '%', $input);
    }

    protected function replaceDateNotation($input)
    {
        if (preg_match('/\d{2}\.\d{2}\.\d{4}/', $input)) {
            $input = implode('-', array_reverse(explode('.', $input)));
        }

        return $input;
    }
}
